<?php
include_once '../../includes/auth.php';
include_once '../../config/database.php';
include_once '../../models/Task.php';
include_once '../../models/User.php';

redirectIfNotLoggedIn();
checkSessionTimeout();

if ($_SESSION['user_role'] !== 'admin') {
    header('Location: ../user/');
    exit;
}

$database = new Database();
$db = $database->getConnection();
$task = new Task($db);
$user = new User($db);

try {
    // Get statistics
    $task_stats = $task->getTaskStatistics();
    $user_stats = $user->getUsersStatistics();
    $pending_approval = $task->getPendingApprovalCount();
    $recent_tasks = $task->getRecentTasks(5);
    $overdue_tasks = $task->getOverdueTasks();
} catch (Exception $e) {
    error_log("Dashboard error: " . $e->getMessage());
    $_SESSION['error'] = 'Unable to load dashboard data. Please try again.';
    $task_stats = ['total_tasks' => 0, 'pending_tasks' => 0, 'in_progress_tasks' => 0, 'completed_tasks' => 0, 'blocked_tasks' => 0];
    $user_stats = ['total_users' => 0, 'admin_users' => 0, 'regular_users' => 0, 'active_users' => 0, 'inactive_users' => 0];
    $pending_approval = 0;
    $recent_tasks = [];
    $overdue_tasks = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Task Management System</title>
    <link rel="stylesheet" href="../../assets/css/style.css">
</head>
<body>
    <div class="app-container">
        <?php include '../../includes/sidebar.php'; ?>
        
        <div class="main-content">
            <div class="main-header">
                <div class="header-content">
                    <h1>Admin Dashboard</h1>
                    <div class="header-actions">
                        <span class="welcome-text">Welcome back, <?php echo htmlspecialchars($_SESSION['user_name']); ?>! 👋</span>
                        <div class="header-buttons">
                            <a href="tasks.php" class="btn btn-primary">
                                <span class="btn-icon">✅</span>
                                Manage Tasks
                            </a>
                            <a href="../user/submit-task.php" class="btn btn-success">
                                <span class="btn-icon">➕</span>
                                Create Task
                            </a>
                        </div>
                    </div>
                </div>
            </div>

            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success">
                    <div class="alert-content">
                        <span class="alert-icon">✅</span>
                        <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-error">
                    <div class="alert-content">
                        <span class="alert-icon">❌</span>
                        <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Statistics Grid -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">📋</div>
                    <div class="stat-value"><?php echo $task_stats['total_tasks']; ?></div>
                    <div class="stat-label">Total Tasks</div>
                    <div class="stat-trend">All Tasks</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">⏳</div>
                    <div class="stat-value"><?php echo $pending_approval; ?></div>
                    <div class="stat-label">Pending Approval</div>
                    <div class="stat-trend">Needs Review</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">🚨</div>
                    <div class="stat-value"><?php echo $overdue_tasks->rowCount(); ?></div>
                    <div class="stat-label">Overdue Tasks</div>
                    <div class="stat-trend">Requires Attention</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">👥</div>
                    <div class="stat-value"><?php echo $user_stats['total_users']; ?></div>
                    <div class="stat-label">Total Users</div>
                    <div class="stat-trend">
                        <?php echo $user_stats['active_users']; ?> Active
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="card">
                <div class="card-header">
                    <h2>Quick Actions</h2>
                    <p class="card-subtitle">Manage your team and tasks efficiently</p>
                </div>
                <div class="card-body">
                    <div class="quick-actions-grid">
                        <a href="tasks.php" class="quick-action-card">
                            <div class="action-icon">✅</div>
                            <div class="action-content">
                                <h3>Manage Tasks</h3>
                                <p>Approve, assign, and track all tasks</p>
                            </div>
                            <div class="action-badge"><?php echo $pending_approval; ?> pending</div>
                        </a>
                        
                        <a href="users.php" class="quick-action-card">
                            <div class="action-icon">👥</div>
                            <div class="action-content">
                                <h3>Manage Users</h3>
                                <p>Add, edit, and manage team members</p>
                            </div>
                            <div class="action-badge"><?php echo $user_stats['total_users']; ?> users</div>
                        </a>
                        
                        <a href="../user/submit-task.php" class="quick-action-card">
                            <div class="action-icon">➕</div>
                            <div class="action-content">
                                <h3>Create Task</h3>
                                <p>Create and assign new tasks</p>
                            </div>
                        </a>
                        
                        <a href="reports.php" class="quick-action-card">
                            <div class="action-icon">📈</div>
                            <div class="action-content">
                                <h3>View Reports</h3>
                                <p>Analytics and performance insights</p>
                            </div>
                        </a>
                    </div>
                </div>
            </div>

            <div class="grid-2-col">
                <!-- Recent Tasks -->
                <div class="card">
                    <div class="card-header">
                        <h2>Recent Tasks</h2>
                        <a href="tasks.php" class="btn btn-secondary">View All</a>
                    </div>
                    <div class="card-body">
                        <?php if ($recent_tasks && $recent_tasks->rowCount() > 0): ?>
                            <div class="task-list">
                                <?php while ($task_item = $recent_tasks->fetch(PDO::FETCH_ASSOC)): ?>
                                <div class="task-list-item">
                                    <div class="task-info">
                                        <h4><?php echo htmlspecialchars($task_item['title']); ?></h4>
                                        <p class="task-meta">
                                            Assigned to: <?php echo htmlspecialchars($task_item['assigned_to_name']); ?>
                                        </p>
                                    </div>
                                    <div class="task-status">
                                        <span class="badge badge-<?php echo getStatusBadgeClass($task_item['status']); ?>">
                                            <?php echo ucfirst($task_item['status']); ?>
                                        </span>
                                        <span class="task-priority badge badge-<?php echo getPriorityBadgeClass($task_item['priority']); ?>">
                                            <?php echo ucfirst($task_item['priority']); ?>
                                        </span>
                                    </div>
                                </div>
                                <?php endwhile; ?>
                            </div>
                        <?php else: ?>
                            <div class="empty-state">
                                <div class="empty-icon">📋</div>
                                <h3>No Tasks Yet</h3>
                                <p>Get started by creating your first task</p>
                                <a href="../user/submit-task.php" class="btn btn-primary">Create Task</a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- System Overview -->
                <div class="card">
                    <div class="card-header">
                        <h2>System Overview</h2>
                    </div>
                    <div class="card-body">
                        <div class="system-stats">
                            <div class="system-stat">
                                <div class="stat-label">Task Completion Rate</div>
                                <div class="stat-progress">
                                    <div class="progress-bar">
                                        <div class="progress-fill" style="width: <?php echo $task_stats['total_tasks'] > 0 ? ($task_stats['completed_tasks'] / $task_stats['total_tasks'] * 100) : 0; ?>%"></div>
                                    </div>
                                    <span class="stat-percentage">
                                        <?php echo $task_stats['total_tasks'] > 0 ? round(($task_stats['completed_tasks'] / $task_stats['total_tasks'] * 100), 1) : 0; ?>%
                                    </span>
                                </div>
                            </div>
                            
                            <div class="system-stat">
                                <div class="stat-label">Active Users</div>
                                <div class="stat-value-large"><?php echo $user_stats['active_users']; ?>/<?php echo $user_stats['total_users']; ?></div>
                            </div>
                            
                            <div class="system-stat">
                                <div class="stat-label">Tasks by Status</div>
                                <div class="status-distribution">
                                    <div class="status-item">
                                        <span class="status-dot pending"></span>
                                        <span>Pending: <?php echo $task_stats['pending_tasks']; ?></span>
                                    </div>
                                    <div class="status-item">
                                        <span class="status-dot in-progress"></span>
                                        <span>In Progress: <?php echo $task_stats['in_progress_tasks']; ?></span>
                                    </div>
                                    <div class="status-item">
                                        <span class="status-dot completed"></span>
                                        <span>Completed: <?php echo $task_stats['completed_tasks']; ?></span>
                                    </div>
                                    <div class="status-item">
                                        <span class="status-dot blocked"></span>
                                        <span>Blocked: <?php echo $task_stats['blocked_tasks']; ?></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Auto-hide alerts after 5 seconds
        setTimeout(() => {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                alert.style.opacity = '0';
                alert.style.transition = 'opacity 0.5s ease';
                setTimeout(() => {
                    alert.style.display = 'none';
                }, 500);
            });
        }, 5000);

        // Add loading states to buttons
        document.querySelectorAll('a.btn').forEach(button => {
            button.addEventListener('click', function(e) {
                const originalText = this.innerHTML;
                this.innerHTML = '<span class="loading-spinner"></span> Loading...';
                this.style.pointerEvents = 'none';
                
                setTimeout(() => {
                    this.innerHTML = originalText;
                    this.style.pointerEvents = 'auto';
                }, 2000);
            });
        });
    </script>
</body>
</html>

<?php
function getStatusBadgeClass($status) {
    switch ($status) {
        case 'completed': return 'success';
        case 'in-progress': return 'info';
        case 'blocked': return 'danger';
        default: return 'warning';
    }
}

function getPriorityBadgeClass($priority) {
    switch ($priority) {
        case 'high': return 'danger';
        case 'medium': return 'warning';
        default: return 'info';
    }
}
?>