<?php
include_once '../../includes/auth.php';
include_once '../../config/database.php';
include_once '../../models/Task.php';
include_once '../../models/User.php';

redirectIfNotLoggedIn();
checkSessionTimeout();

if ($_SESSION['user_role'] !== 'admin') {
    header('Location: ../user/');
    exit;
}

$database = new Database();
$db = $database->getConnection();
$task = new Task($db);
$user = new User($db);

// Get statistics
try {
    $task_stats = $task->getTaskStatistics();
    $user_stats = $user->getUsersStatistics();
    $pending_approval = $task->getPendingApprovalCount();
    $overdue_tasks = $task->getOverdueTasks();
    $recent_tasks = $task->getRecentTasks(10);
    
    // Calculate completion rate
    $completion_rate = $task_stats['total_tasks'] > 0 ? 
        round(($task_stats['completed_tasks'] / $task_stats['total_tasks']) * 100, 1) : 0;
        
    // Get tasks by priority
    $high_priority_tasks = $task->getTasksByPriority('high');
    $medium_priority_tasks = $task->getTasksByPriority('medium');
    $low_priority_tasks = $task->getTasksByPriority('low');
} catch (Exception $e) {
    error_log("Reports page error: " . $e->getMessage());
    $_SESSION['error'] = 'Unable to load reports data. Please try again.';
    $task_stats = ['total_tasks' => 0, 'pending_tasks' => 0, 'in_progress_tasks' => 0, 'completed_tasks' => 0, 'blocked_tasks' => 0];
    $user_stats = ['total_users' => 0, 'admin_users' => 0, 'regular_users' => 0, 'active_users' => 0, 'inactive_users' => 0];
    $pending_approval = 0;
    $completion_rate = 0;
    $overdue_tasks = [];
    $recent_tasks = [];
    $high_priority_tasks = $medium_priority_tasks = $low_priority_tasks = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports & Analytics - Admin</title>
    <link rel="stylesheet" href="../../assets/css/style.css">
</head>
<body>
    <div class="app-container">
        <?php include '../../includes/sidebar.php'; ?>
        
        <div class="main-content">
            <div class="main-header">
                <div class="header-content">
                    <h1>Reports & Analytics</h1>
                    <div class="header-actions">
                        <button class="btn btn-secondary" onclick="window.print()">
                            <span class="btn-icon">🖨️</span>
                            Print Report
                        </button>
                        <button class="btn btn-success" onclick="exportToPDF()">
                            <span class="btn-icon">📊</span>
                            Export PDF
                        </button>
                    </div>
                </div>
            </div>

            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success">
                    <div class="alert-content">
                        <span class="alert-icon">✅</span>
                        <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-error">
                    <div class="alert-content">
                        <span class="alert-icon">❌</span>
                        <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Key Metrics -->
            <div class="stats-grid">
                <div class="stat-card highlight">
                    <div class="stat-icon">📈</div>
                    <div class="stat-value"><?php echo $completion_rate; ?>%</div>
                    <div class="stat-label">Completion Rate</div>
                    <div class="stat-trend">Overall Performance</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">✅</div>
                    <div class="stat-value"><?php echo $task_stats['completed_tasks']; ?></div>
                    <div class="stat-label">Tasks Completed</div>
                    <div class="stat-trend"><?php echo $task_stats['total_tasks']; ?> total</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">👥</div>
                    <div class="stat-value"><?php echo $user_stats['active_users']; ?></div>
                    <div class="stat-label">Active Users</div>
                    <div class="stat-trend"><?php echo $user_stats['total_users']; ?> registered</div>
                </div>
                
                <div class="stat-card warning">
                    <div class="stat-icon">🚨</div>
                    <div class="stat-value"><?php echo $overdue_tasks->rowCount(); ?></div>
                    <div class="stat-label">Overdue Tasks</div>
                    <div class="stat-trend">Requires Attention</div>
                </div>
            </div>

            <div class="grid-2-col">
                <!-- Task Distribution -->
                <div class="card">
                    <div class="card-header">
                        <h2>Task Status Distribution</h2>
                    </div>
                    <div class="card-body">
                        <div class="chart-container">
                            <div class="status-chart">
                                <?php
                                $statuses = [
                                    'completed' => $task_stats['completed_tasks'],
                                    'in-progress' => $task_stats['in_progress_tasks'],
                                    'pending' => $task_stats['pending_tasks'],
                                    'blocked' => $task_stats['blocked_tasks']
                                ];
                                $total = array_sum($statuses);
                                
                                foreach ($statuses as $status => $count):
                                    if ($total > 0):
                                        $percentage = round(($count / $total) * 100, 1);
                                ?>
                                <div class="chart-item">
                                    <div class="chart-label">
                                        <span class="status-dot <?php echo $status; ?>"></span>
                                        <span><?php echo ucfirst($status); ?></span>
                                        <span class="chart-value"><?php echo $count; ?> (<?php echo $percentage; ?>%)</span>
                                    </div>
                                    <div class="chart-bar">
                                        <div class="chart-fill <?php echo $status; ?>" style="width: <?php echo $percentage; ?>%"></div>
                                    </div>
                                </div>
                                <?php
                                    endif;
                                endforeach;
                                ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- User Statistics -->
                <div class="card">
                    <div class="card-header">
                        <h2>User Statistics</h2>
                    </div>
                    <div class="card-body">
                        <div class="user-stats-grid">
                            <div class="user-stat">
                                <div class="stat-icon">👑</div>
                                <div class="stat-content">
                                    <div class="stat-number"><?php echo $user_stats['admin_users']; ?></div>
                                    <div class="stat-label">Administrators</div>
                                </div>
                            </div>
                            <div class="user-stat">
                                <div class="stat-icon">👤</div>
                                <div class="stat-content">
                                    <div class="stat-number"><?php echo $user_stats['regular_users']; ?></div>
                                    <div class="stat-label">Regular Users</div>
                                </div>
                            </div>
                            <div class="user-stat">
                                <div class="stat-icon">✅</div>
                                <div class="stat-content">
                                    <div class="stat-number"><?php echo $user_stats['active_users']; ?></div>
                                    <div class="stat-label">Active Users</div>
                                </div>
                            </div>
                            <div class="user-stat">
                                <div class="stat-icon">❌</div>
                                <div class="stat-content">
                                    <div class="stat-number"><?php echo $user_stats['inactive_users']; ?></div>
                                    <div class="stat-label">Inactive Users</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="progress-section">
                            <h4>User Activity Rate</h4>
                            <div class="progress-bar large">
                                <div class="progress-fill" style="width: <?php echo $user_stats['total_users'] > 0 ? ($user_stats['active_users'] / $user_stats['total_users'] * 100) : 0; ?>%"></div>
                            </div>
                            <div class="progress-label">
                                <span><?php echo $user_stats['active_users']; ?> active of <?php echo $user_stats['total_users']; ?> total</span>
                                <span><?php echo $user_stats['total_users'] > 0 ? round(($user_stats['active_users'] / $user_stats['total_users'] * 100), 1) : 0; ?>%</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Overdue Tasks -->
            <div class="card">
                <div class="card-header">
                    <h2>🚨 Overdue Tasks</h2>
                    <span class="badge badge-danger"><?php echo $overdue_tasks->rowCount(); ?> overdue</span>
                </div>
                <div class="card-body">
                    <?php if ($overdue_tasks->rowCount() > 0): ?>
                        <div class="table-responsive">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>Task Title</th>
                                        <th>Assigned To</th>
                                        <th>Due Date</th>
                                        <th>Days Overdue</th>
                                        <th>Priority</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while ($task_item = $overdue_tasks->fetch(PDO::FETCH_ASSOC)): 
                                        $days_overdue = floor((time() - strtotime($task_item['due_date'])) / (60 * 60 * 24));
                                    ?>
                                    <tr>
                                        <td>
                                            <div class="task-title"><?php echo htmlspecialchars($task_item['title']); ?></div>
                                            <div class="task-description"><?php echo htmlspecialchars(substr($task_item['description'], 0, 100)); ?>...</div>
                                        </td>
                                        <td><?php echo htmlspecialchars($task_item['assigned_to_name']); ?></td>
                                        <td>
                                            <span class="date-badge overdue">
                                                <?php echo date('M j, Y', strtotime($task_item['due_date'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge badge-danger"><?php echo $days_overdue; ?> days</span>
                                        </td>
                                        <td>
                                            <span class="badge badge-<?php echo getPriorityBadgeClass($task_item['priority']); ?>">
                                                <?php echo ucfirst($task_item['priority']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge badge-<?php echo getStatusBadgeClass($task_item['status']); ?>">
                                                <?php echo ucfirst($task_item['status']); ?>
                                            </span>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="empty-state">
                            <div class="empty-icon">🎉</div>
                            <h3>No Overdue Tasks</h3>
                            <p>Excellent! All tasks are up to date.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Recent Activity -->
            <div class="card">
                <div class="card-header">
                    <h2>Recent Activity</h2>
                    <a href="tasks.php" class="btn btn-secondary">View All Tasks</a>
                </div>
                <div class="card-body">
                    <?php if ($recent_tasks && $recent_tasks->rowCount() > 0): ?>
                        <div class="activity-timeline">
                            <?php while ($task_item = $recent_tasks->fetch(PDO::FETCH_ASSOC)): ?>
                            <div class="timeline-item">
                                <div class="timeline-marker"></div>
                                <div class="timeline-content">
                                    <div class="timeline-header">
                                        <h4><?php echo htmlspecialchars($task_item['title']); ?></h4>
                                        <span class="timeline-time"><?php echo timeAgo($task_item['created_at']); ?></span>
                                    </div>
                                    <div class="timeline-body">
                                        <p>Assigned to <strong><?php echo htmlspecialchars($task_item['assigned_to_name']); ?></strong></p>
                                        <div class="timeline-meta">
                                            <span class="badge badge-<?php echo getStatusBadgeClass($task_item['status']); ?>">
                                                <?php echo ucfirst($task_item['status']); ?>
                                            </span>
                                            <span class="badge badge-<?php echo getPriorityBadgeClass($task_item['priority']); ?>">
                                                <?php echo ucfirst($task_item['priority']); ?>
                                            </span>
                                            <span class="due-date">Due: <?php echo date('M j, Y', strtotime($task_item['due_date'])); ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endwhile; ?>
                        </div>
                    <?php else: ?>
                        <div class="empty-state">
                            <div class="empty-icon">📋</div>
                            <h3>No Recent Activity</h3>
                            <p>There hasn't been any task activity recently.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <style>
    .stat-card.highlight {
        background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
        color: white;
    }

    .stat-card.highlight .stat-value,
    .stat-card.highlight .stat-label,
    .stat-card.highlight .stat-trend {
        color: white;
    }

    .stat-card.warning {
        border-left: 4px solid var(--warning);
    }

    .chart-container {
        padding: 1rem 0;
    }

    .chart-item {
        margin-bottom: 1rem;
    }

    .chart-label {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 0.5rem;
        font-size: 0.9rem;
    }

    .chart-value {
        color: var(--text-muted);
        font-weight: 600;
    }

    .chart-bar {
        height: 8px;
        background: var(--bg-tertiary);
        border-radius: 4px;
        overflow: hidden;
    }

    .chart-fill {
        height: 100%;
        border-radius: 4px;
        transition: width 0.5s ease;
    }

    .chart-fill.completed { background: var(--success); }
    .chart-fill.in-progress { background: var(--secondary); }
    .chart-fill.pending { background: var(--warning); }
    .chart-fill.blocked { background: var(--danger); }

    .user-stats-grid {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
        margin-bottom: 2rem;
    }

    .user-stat {
        display: flex;
        align-items: center;
        gap: 1rem;
        padding: 1rem;
        background: var(--bg-tertiary);
        border-radius: var(--radius);
    }

    .user-stat .stat-icon {
        font-size: 1.5rem;
    }

    .stat-number {
        font-size: 1.5rem;
        font-weight: 700;
        color: var(--text-primary);
    }

    .stat-label {
        color: var(--text-secondary);
        font-size: 0.9rem;
    }

    .progress-section {
        margin-top: 1.5rem;
    }

    .progress-bar.large {
        height: 12px;
    }

    .progress-label {
        display: flex;
        justify-content: space-between;
        margin-top: 0.5rem;
        font-size: 0.9rem;
        color: var(--text-muted);
    }

    .activity-timeline {
        position: relative;
    }

    .timeline-item {
        display: flex;
        margin-bottom: 1.5rem;
        position: relative;
    }

    .timeline-marker {
        width: 12px;
        height: 12px;
        background: var(--primary);
        border-radius: 50%;
        margin-right: 1rem;
        margin-top: 0.25rem;
        flex-shrink: 0;
    }

    .timeline-content {
        flex: 1;
        padding-bottom: 1.5rem;
        border-left: 2px solid var(--border);
        padding-left: 1rem;
    }

    .timeline-header {
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        margin-bottom: 0.5rem;
    }

    .timeline-header h4 {
        margin: 0;
        color: var(--text-primary);
        font-size: 1rem;
    }

    .timeline-time {
        color: var(--text-muted);
        font-size: 0.8rem;
        white-space: nowrap;
    }

    .timeline-body p {
        margin: 0 0 0.5rem 0;
        color: var(--text-secondary);
    }

    .timeline-meta {
        display: flex;
        gap: 0.5rem;
        flex-wrap: wrap;
    }

    .due-date {
        color: var(--text-muted);
        font-size: 0.8rem;
    }

    @media print {
        .sidebar, .header-actions {
            display: none !important;
        }
        
        .main-content {
            margin-left: 0 !important;
        }
        
        .card {
            break-inside: avoid;
        }
    }
    </style>

    <script>
    function exportToPDF() {
        // Show loading state
        const btn = event.target;
        const originalText = btn.innerHTML;
        btn.innerHTML = '<span class="loading-spinner"></span> Generating PDF...';
        btn.disabled = true;
        
        // Simulate PDF generation
        setTimeout(() => {
            btn.innerHTML = originalText;
            btn.disabled = false;
            alert('PDF export functionality would be implemented here. In a real application, this would generate a downloadable PDF report.');
        }, 2000);
    }

    // Auto-hide alerts
    setTimeout(() => {
        const alerts = document.querySelectorAll('.alert');
        alerts.forEach(alert => {
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.style.display = 'none';
            }, 500);
        });
    }, 5000);

    // Animate chart bars on page load
    document.addEventListener('DOMContentLoaded', function() {
        const chartFills = document.querySelectorAll('.chart-fill');
        chartFills.forEach(fill => {
            const originalWidth = fill.style.width;
            fill.style.width = '0';
            setTimeout(() => {
                fill.style.width = originalWidth;
            }, 500);
        });
    });
    </script>
</body>
</html>

<?php
function getStatusBadgeClass($status) {
    switch ($status) {
        case 'completed': return 'success';
        case 'in-progress': return 'info';
        case 'blocked': return 'danger';
        default: return 'warning';
    }
}

function getPriorityBadgeClass($priority) {
    switch ($priority) {
        case 'high': return 'danger';
        case 'medium': return 'warning';
        default: return 'info';
    }
}

function timeAgo($datetime) {
    $time = strtotime($datetime);
    $now = time();
    $diff = $now - $time;
    
    if ($diff < 60) {
        return 'Just now';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' min' . ($mins > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
    } elseif ($diff < 2592000) {
        $days = floor($diff / 86400);
        return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
    } else {
        return date('M j, Y', $time);
    }
}
?>