<?php
include_once '../../includes/auth.php';
include_once '../../config/database.php';
include_once '../../models/Task.php';
include_once '../../models/User.php';

redirectIfNotLoggedIn();
checkSessionTimeout();

if ($_SESSION['user_role'] !== 'admin') {
    header('Location: ../user/');
    exit;
}

$database = new Database();
$db = $database->getConnection();
$task = new Task($db);
$user = new User($db);

// Handle AJAX request for task details
if (isset($_GET['action']) && $_GET['action'] === 'get_task_details') {
    header('Content-Type: application/json');
    
    try {
        $task_id = $_GET['task_id'] ?? 0;
        $task_details = $task->getTask($task_id);
        
        if ($task_details) {
            // Get task comments
            $comments_result = $task->getComments($task_id);
            $comments = $comments_result ? $comments_result->fetchAll(PDO::FETCH_ASSOC) : [];
            
            echo json_encode([
                'success' => true,
                'task' => $task_details,
                'comments' => $comments
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Task not found'
            ]);
        }
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error loading task details: ' . $e->getMessage()
        ]);
    }
    exit;
}

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        switch ($action) {
            case 'approve_task':
                if (isset($_POST['task_id'])) {
                    if ($task->approveTask($_POST['task_id'], $_SESSION['user_id'])) {
                        $_SESSION['success'] = 'Task approved successfully.';
                    } else {
                        $_SESSION['error'] = 'Failed to approve task.';
                    }
                }
                break;
                
            case 'reject_task':
                if (isset($_POST['task_id'])) {
                    if ($task->rejectTask($_POST['task_id'])) {
                        $_SESSION['success'] = 'Task rejected successfully.';
                    } else {
                        $_SESSION['error'] = 'Failed to reject task.';
                    }
                }
                break;
                
            case 'delete_task':
                if (isset($_POST['task_id'])) {
                    if ($task->deleteTask($_POST['task_id'])) {
                        $_SESSION['success'] = 'Task deleted successfully.';
                    } else {
                        $_SESSION['error'] = 'Failed to delete task.';
                    }
                }
                break;

            case 'update_status':
                if (isset($_POST['task_id']) && isset($_POST['status'])) {
                    $comment = $_POST['comment'] ?? '';
                    if ($task->updateStatus($_POST['task_id'], $_POST['status'], $comment, $_SESSION['user_id'])) {
                        $_SESSION['success'] = 'Task status updated successfully.';
                    } else {
                        $_SESSION['error'] = 'Failed to update task status.';
                    }
                }
                break;

            case 'reassign_task':
                if (isset($_POST['task_id']) && isset($_POST['assigned_to'])) {
                    if ($task->reassignTask($_POST['task_id'], $_POST['assigned_to'], $_SESSION['user_id'])) {
                        $_SESSION['success'] = 'Task reassigned successfully.';
                    } else {
                        $_SESSION['error'] = 'Failed to reassign task.';
                    }
                }
                break;

            case 'add_comment':
                if (isset($_POST['task_id']) && isset($_POST['comment'])) {
                    if ($task->addComment($_POST['task_id'], $_SESSION['user_id'], $_POST['comment'])) {
                        $_SESSION['success'] = 'Comment added successfully.';
                    } else {
                        $_SESSION['error'] = 'Failed to add comment.';
                    }
                }
                break;
        }
    } catch (Exception $e) {
        $_SESSION['error'] = $e->getMessage();
    }
    
    header('Location: tasks.php');
    exit;
}

// Get all tasks and data
try {
    // Get all tasks (admin sees all including unapproved and rejected)
    $all_tasks_result = $task->readAll($_SESSION['user_id'], 'admin');
    $all_tasks = $all_tasks_result ? $all_tasks_result->fetchAll(PDO::FETCH_ASSOC) : [];
    
    // Get pending approval tasks
    $pending_tasks_result = $task->getPendingApprovalTasks();
    $pending_tasks = $pending_tasks_result ? $pending_tasks_result->fetchAll(PDO::FETCH_ASSOC) : [];
    
    // Get task statistics
    $task_stats = $task->getTaskStatistics();
    
    // Get overdue tasks
    $overdue_tasks_result = $task->getOverdueTasks();
    $overdue_tasks = $overdue_tasks_result ? $overdue_tasks_result->fetchAll(PDO::FETCH_ASSOC) : [];
    
    // Get all users for reassignment
    $users_result = $user->getAllUsers();
    $users = $users_result ? $users_result->fetchAll(PDO::FETCH_ASSOC) : [];
    
} catch (Exception $e) {
    error_log("Tasks page error: " . $e->getMessage());
    $_SESSION['error'] = 'Unable to load tasks. Please try again. Error: ' . $e->getMessage();
    $all_tasks = [];
    $pending_tasks = [];
    $task_stats = ['total_tasks' => 0, 'pending_tasks' => 0, 'in_progress_tasks' => 0, 'completed_tasks' => 0, 'blocked_tasks' => 0];
    $overdue_tasks = [];
    $users = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Tasks - Admin</title>
    <link rel="stylesheet" href="../../assets/css/style.css">
</head>
<body>
    <div class="app-container">
        <?php include '../../includes/sidebar.php'; ?>
        
        <div class="main-content">
            <div class="main-header">
                <div class="header-content">
                    <h1>Manage Tasks</h1>
                    <div class="header-actions">
                        <a href="../user/submit-task.php" class="btn btn-success">
                            <span class="btn-icon">➕</span>
                            Create New Task
                        </a>
                        <button type="button" class="btn btn-info" onclick="refreshPage()">
                            <span class="btn-icon">🔄</span>
                            Refresh
                        </button>
                    </div>
                </div>
            </div>

            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success">
                    <div class="alert-content">
                        <span class="alert-icon">✅</span>
                        <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-error">
                    <div class="alert-content">
                        <span class="alert-icon">❌</span>
                        <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Task Statistics -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon primary">📋</div>
                    <div class="stat-info">
                        <div class="stat-number"><?php echo count($all_tasks); ?></div>
                        <div class="stat-label">Total Tasks</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon warning">⏳</div>
                    <div class="stat-info">
                        <div class="stat-number"><?php echo count($pending_tasks); ?></div>
                        <div class="stat-label">Pending Approval</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon danger">⚠️</div>
                    <div class="stat-info">
                        <div class="stat-number"><?php echo count($overdue_tasks); ?></div>
                        <div class="stat-label">Overdue</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon success">✅</div>
                    <div class="stat-info">
                        <div class="stat-number"><?php echo $task_stats['completed_tasks']; ?></div>
                        <div class="stat-label">Completed</div>
                    </div>
                </div>
            </div>

            <!-- Pending Approval Tasks -->
            <?php if (!empty($pending_tasks)): ?>
            <div class="card">
                <div class="card-header">
                    <h2>📋 Tasks Pending Approval</h2>
                    <span class="badge badge-warning"><?php echo count($pending_tasks); ?> pending</span>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Title & Description</th>
                                    <th>Submitted By</th>
                                    <th>Assigned To</th>
                                    <th>Due Date</th>
                                    <th>Priority</th>
                                    <th>Created Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($pending_tasks as $task_item): ?>
                                <tr>
                                    <td>
                                        <div class="task-title"><?php echo htmlspecialchars($task_item['title']); ?></div>
                                        <div class="task-description"><?php echo htmlspecialchars(substr($task_item['description'] ?? '', 0, 100)); ?>...</div>
                                    </td>
                                    <td>
                                        <div class="user-info-small">
                                            <div class="user-name"><?php echo htmlspecialchars($task_item['submitted_by_name'] ?? 'Unknown'); ?></div>
                                            <div class="user-email"><?php echo htmlspecialchars($task_item['submitted_by'] ?? ''); ?></div>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="user-info-small">
                                            <div class="user-name"><?php echo htmlspecialchars($task_item['assigned_to_name'] ?? 'Unassigned'); ?></div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="date-badge <?php echo isOverdue($task_item['due_date']) ? 'overdue' : ''; ?>">
                                            <?php echo date('M j, Y', strtotime($task_item['due_date'])); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge badge-<?php echo getPriorityBadgeClass($task_item['priority']); ?>">
                                            <?php echo ucfirst($task_item['priority']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php echo date('M j, Y', strtotime($task_item['created_at'])); ?>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <form method="POST" class="inline-form">
                                                <input type="hidden" name="action" value="approve_task">
                                                <input type="hidden" name="task_id" value="<?php echo $task_item['id']; ?>">
                                                <button type="submit" class="btn btn-success btn-sm" title="Approve Task">
                                                    ✅ Approve
                                                </button>
                                            </form>
                                            <form method="POST" class="inline-form">
                                                <input type="hidden" name="action" value="reject_task">
                                                <input type="hidden" name="task_id" value="<?php echo $task_item['id']; ?>">
                                                <button type="submit" class="btn btn-danger btn-sm" 
                                                        onclick="return confirm('Are you sure you want to reject this task?')"
                                                        title="Reject Task">
                                                    ❌ Reject
                                                </button>
                                            </form>
                                            <button type="button" class="btn btn-info btn-sm" 
                                                    onclick="viewTaskDetails(<?php echo $task_item['id']; ?>)"
                                                    title="View Details">
                                                👁️ View
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <!-- All Tasks -->
            <div class="card">
                <div class="card-header">
                    <div class="header-with-actions">
                        <div>
                            <h2>📊 All Tasks</h2>
                            <span class="badge badge-info"><?php echo count($all_tasks); ?> total tasks</span>
                        </div>
                        <div class="export-actions">
                            <button type="button" class="btn btn-secondary btn-sm" onclick="exportTasks()">
                                📥 Export Tasks
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-body">
                    <?php if (!empty($all_tasks)): ?>
                        <!-- Task Filters -->
                        <div class="filters-bar">
                            <div class="filter-group">
                                <label for="statusFilter">Status:</label>
                                <select id="statusFilter" class="filter-select" onchange="filterTasks()">
                                    <option value="">All Status</option>
                                    <option value="pending">Pending</option>
                                    <option value="in-progress">In Progress</option>
                                    <option value="completed">Completed</option>
                                    <option value="blocked">Blocked</option>
                                    <option value="rejected">Rejected</option>
                                </select>
                            </div>
                            <div class="filter-group">
                                <label for="priorityFilter">Priority:</label>
                                <select id="priorityFilter" class="filter-select" onchange="filterTasks()">
                                    <option value="">All Priority</option>
                                    <option value="low">Low</option>
                                    <option value="medium">Medium</option>
                                    <option value="high">High</option>
                                </select>
                            </div>
                            <div class="filter-group">
                                <label for="approvalFilter">Approval:</label>
                                <select id="approvalFilter" class="filter-select" onchange="filterTasks()">
                                    <option value="">All</option>
                                    <option value="approved">Approved</option>
                                    <option value="pending">Pending Approval</option>
                                </select>
                            </div>
                            <div class="filter-group">
                                <label for="searchFilter">Search:</label>
                                <input type="text" id="searchFilter" class="filter-select" placeholder="Search tasks..." onkeyup="searchTasks()">
                            </div>
                            <div class="filter-group">
                                <button type="button" class="btn btn-secondary btn-sm" onclick="resetFilters()">
                                    🔄 Reset
                                </button>
                            </div>
                        </div>

                        <div class="table-responsive">
                            <table class="data-table" id="tasksTable">
                                <thead>
                                    <tr>
                                        <th>Title</th>
                                        <th>Assigned To</th>
                                        <th>Submitted By</th>
                                        <th>Status</th>
                                        <th>Due Date</th>
                                        <th>Priority</th>
                                        <th>Approval</th>
                                        <th>Created</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($all_tasks as $task_item): ?>
                                    <tr class="task-row" 
                                        data-status="<?php echo $task_item['status']; ?>"
                                        data-priority="<?php echo $task_item['priority']; ?>"
                                        data-approved="<?php echo $task_item['is_approved'] ? 'approved' : 'pending'; ?>"
                                        data-search="<?php echo htmlspecialchars(strtolower($task_item['title'] . ' ' . $task_item['description'])); ?>">
                                        <td>
                                            <div class="task-title"><?php echo htmlspecialchars($task_item['title']); ?></div>
                                            <div class="task-description"><?php echo htmlspecialchars(substr($task_item['description'] ?? '', 0, 80)); ?>...</div>
                                        </td>
                                        <td>
                                            <div class="user-info-small">
                                                <div class="user-name"><?php echo htmlspecialchars($task_item['assigned_to_name'] ?? 'Unassigned'); ?></div>
                                                <div class="user-role"><?php echo ucfirst($task_item['assigned_to_role'] ?? 'Unknown'); ?></div>
                                            </div>
                                        </td>
                                        <td>
                                            <div class="user-info-small">
                                                <div class="user-name"><?php echo htmlspecialchars($task_item['submitted_by_name'] ?? 'Unknown'); ?></div>
                                                <div class="user-email"><?php echo date('M j, Y', strtotime($task_item['created_at'])); ?></div>
                                            </div>
                                        </td>
                                        <td>
                                            <span class="badge badge-<?php echo getStatusBadgeClass($task_item['status']); ?>">
                                                <?php echo ucfirst($task_item['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="date-badge <?php echo isOverdue($task_item['due_date']) ? 'overdue' : ''; ?>">
                                                <?php echo date('M j, Y', strtotime($task_item['due_date'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge badge-<?php echo getPriorityBadgeClass($task_item['priority']); ?>">
                                                <?php echo ucfirst($task_item['priority']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($task_item['is_approved']): ?>
                                                <span class="badge badge-success">Approved</span>
                                                <?php if ($task_item['approved_by_name']): ?>
                                                    <div class="user-info-small">
                                                        <div class="user-name">by <?php echo htmlspecialchars($task_item['approved_by_name']); ?></div>
                                                        <div class="user-email"><?php echo $task_item['approved_at'] ? date('M j, Y', strtotime($task_item['approved_at'])) : ''; ?></div>
                                                    </div>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                <span class="badge badge-warning">Pending</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php echo date('M j, Y', strtotime($task_item['created_at'])); ?>
                                        </td>
                                        <td>
                                            <div class="action-buttons">
                                                <button type="button" class="btn btn-secondary btn-sm" 
                                                        onclick="viewTaskDetails(<?php echo $task_item['id']; ?>)"
                                                        title="View Details">
                                                    👁️
                                                </button>
                                                
                                                <!-- Status Update Dropdown -->
                                                <div class="dropdown">
                                                    <button class="btn btn-info btn-sm dropdown-toggle" type="button">
                                                        📝
                                                    </button>
                                                    <div class="dropdown-menu">
                                                        <form method="POST" class="inline-form">
                                                            <input type="hidden" name="action" value="update_status">
                                                            <input type="hidden" name="task_id" value="<?php echo $task_item['id']; ?>">
                                                            <input type="hidden" name="status" value="pending">
                                                            <button type="submit" class="dropdown-item">🟡 Pending</button>
                                                        </form>
                                                        <form method="POST" class="inline-form">
                                                            <input type="hidden" name="action" value="update_status">
                                                            <input type="hidden" name="task_id" value="<?php echo $task_item['id']; ?>">
                                                            <input type="hidden" name="status" value="in-progress">
                                                            <button type="submit" class="dropdown-item">🔵 In Progress</button>
                                                        </form>
                                                        <form method="POST" class="inline-form">
                                                            <input type="hidden" name="action" value="update_status">
                                                            <input type="hidden" name="task_id" value="<?php echo $task_item['id']; ?>">
                                                            <input type="hidden" name="status" value="completed">
                                                            <button type="submit" class="dropdown-item">✅ Completed</button>
                                                        </form>
                                                        <form method="POST" class="inline-form">
                                                            <input type="hidden" name="action" value="update_status">
                                                            <input type="hidden" name="task_id" value="<?php echo $task_item['id']; ?>">
                                                            <input type="hidden" name="status" value="blocked">
                                                            <button type="submit" class="dropdown-item">🔴 Blocked</button>
                                                        </form>
                                                    </div>
                                                </div>

                                                <!-- Reassign Dropdown -->
                                                <?php if (!empty($users)): ?>
                                                <div class="dropdown">
                                                    <button class="btn btn-warning btn-sm dropdown-toggle" type="button">
                                                        🔄
                                                    </button>
                                                    <div class="dropdown-menu">
                                                        <?php foreach ($users as $user_item): ?>
                                                            <form method="POST" class="inline-form">
                                                                <input type="hidden" name="action" value="reassign_task">
                                                                <input type="hidden" name="task_id" value="<?php echo $task_item['id']; ?>">
                                                                <input type="hidden" name="assigned_to" value="<?php echo $user_item['id']; ?>">
                                                                <button type="submit" class="dropdown-item">
                                                                    <?php echo htmlspecialchars($user_item['name']); ?> (<?php echo ucfirst($user_item['role']); ?>)
                                                                </button>
                                                            </form>
                                                        <?php endforeach; ?>
                                                    </div>
                                                </div>
                                                <?php endif; ?>

                                                <?php if (!$task_item['is_approved']): ?>
                                                    <form method="POST" class="inline-form">
                                                        <input type="hidden" name="action" value="approve_task">
                                                        <input type="hidden" name="task_id" value="<?php echo $task_item['id']; ?>">
                                                        <button type="submit" class="btn btn-success btn-sm" title="Approve Task">
                                                            ✅
                                                        </button>
                                                    </form>
                                                <?php endif; ?>
                                                
                                                <form method="POST" class="inline-form">
                                                    <input type="hidden" name="action" value="delete_task">
                                                    <input type="hidden" name="task_id" value="<?php echo $task_item['id']; ?>">
                                                    <button type="submit" class="btn btn-danger btn-sm" 
                                                            onclick="return confirm('Are you sure you want to delete this task? This action cannot be undone.')"
                                                            title="Delete Task">
                                                        🗑️
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Tasks Summary -->
                        <div class="tasks-summary">
                            <div class="summary-item">
                                <strong>Showing:</strong> <span id="visibleTasksCount"><?php echo count($all_tasks); ?></span> of <?php echo count($all_tasks); ?> tasks
                            </div>
                        </div>

                    <?php else: ?>
                        <div class="empty-state">
                            <div class="empty-icon">📋</div>
                            <h3>No Tasks Found</h3>
                            <p>There are no tasks in the system yet.</p>
                            <a href="../user/submit-task.php" class="btn btn-primary">Create First Task</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Task Details Modal -->
    <div id="taskModal" class="modal">
        <div class="modal-content large">
            <div class="modal-header">
                <h3>Task Details</h3>
                <button type="button" class="close-btn" onclick="closeTaskModal()">&times;</button>
            </div>
            <div class="modal-body" id="taskModalBody">
                <!-- Content will be loaded dynamically -->
            </div>
        </div>
    </div>

    <style>
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: var(--bg-secondary);
        border-radius: var(--radius-md);
        padding: 1.5rem;
        display: flex;
        align-items: center;
        gap: 1rem;
        border: 1px solid var(--border-color);
    }

    .stat-icon {
        font-size: 2rem;
        padding: 1rem;
        border-radius: var(--radius-md);
    }

    .stat-icon.primary { background: var(--primary-light); color: var(--primary); }
    .stat-icon.warning { background: var(--warning-light); color: var(--warning); }
    .stat-icon.danger { background: var(--danger-light); color: var(--danger); }
    .stat-icon.success { background: var(--success-light); color: var(--success); }

    .stat-number {
        font-size: 1.5rem;
        font-weight: 700;
        color: var(--text-primary);
    }

    .stat-label {
        color: var(--text-muted);
        font-size: 0.9rem;
    }

    .filters-bar {
        display: flex;
        gap: 1rem;
        margin-bottom: 1.5rem;
        padding: 1rem;
        background: var(--bg-tertiary);
        border-radius: var(--radius-md);
        flex-wrap: wrap;
        align-items: end;
    }

    .filter-group {
        display: flex;
        flex-direction: column;
        gap: 0.5rem;
    }

    .filter-group label {
        font-weight: 500;
        color: var(--text-primary);
        font-size: 0.9rem;
    }

    .filter-select {
        padding: 0.5rem;
        border: 1px solid var(--border-color);
        border-radius: var(--radius-sm);
        background: var(--bg-primary);
        color: var(--text-primary);
    }

    .header-with-actions {
        display: flex;
        justify-content: space-between;
        align-items: center;
        width: 100%;
    }

    .export-actions {
        display: flex;
        gap: 0.5rem;
    }

    .tasks-summary {
        margin-top: 1rem;
        padding: 1rem;
        background: var(--bg-tertiary);
        border-radius: var(--radius-md);
        font-size: 0.9rem;
    }

    .dropdown {
        position: relative;
        display: inline-block;
    }

    .dropdown-toggle::after {
        content: ' ▼';
        font-size: 0.7em;
    }

    .dropdown-menu {
        display: none;
        position: absolute;
        background: white;
        border: 1px solid var(--border-color);
        border-radius: var(--radius-sm);
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        z-index: 1000;
        min-width: 200px;
        max-height: 300px;
        overflow-y: auto;
    }

    .dropdown:hover .dropdown-menu {
        display: block;
    }

    .dropdown-item {
        display: block;
        width: 100%;
        padding: 0.5rem 1rem;
        border: none;
        background: none;
        text-align: left;
        cursor: pointer;
        border-bottom: 1px solid var(--border-color);
    }

    .dropdown-item:hover {
        background: var(--bg-tertiary);
    }

    .dropdown-item:last-child {
        border-bottom: none;
    }

    .table-responsive {
        overflow-x: auto;
    }

    .task-title {
        font-weight: 600;
        color: var(--text-primary);
        margin-bottom: 0.25rem;
    }

    .task-description {
        color: var(--text-muted);
        font-size: 0.85rem;
        line-height: 1.4;
    }

    .user-info-small .user-name {
        font-weight: 500;
        color: var(--text-primary);
    }

    .user-info-small .user-email,
    .user-info-small .user-role {
        color: var(--text-muted);
        font-size: 0.8rem;
    }

    .date-badge {
        padding: 0.3rem 0.6rem;
        border-radius: var(--radius-sm);
        font-size: 0.8rem;
        font-weight: 500;
        background: var(--bg-tertiary);
        color: var(--text-secondary);
    }

    .date-badge.overdue {
        background: var(--danger);
        color: white;
    }

    .action-buttons {
        display: flex;
        gap: 0.3rem;
        flex-wrap: wrap;
    }

    .inline-form {
        display: inline;
    }

    .btn-sm {
        padding: 0.4rem 0.6rem;
        font-size: 0.75rem;
    }

    .modal.large {
        max-width: 800px;
    }

    .task-row.hidden {
        display: none;
    }

    /* Task Details Styles */
    .task-details-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 2rem;
        margin-bottom: 2rem;
    }

    .task-info-section, .task-assignment-section {
        background: var(--bg-secondary);
        padding: 1.5rem;
        border-radius: var(--radius-md);
        border: 1px solid var(--border-color);
    }

    .task-info-section h4, .task-assignment-section h4 {
        margin-top: 0;
        margin-bottom: 1rem;
        color: var(--text-primary);
        border-bottom: 2px solid var(--primary);
        padding-bottom: 0.5rem;
    }

    .info-item {
        display: flex;
        justify-content: space-between;
        margin-bottom: 0.75rem;
        padding-bottom: 0.75rem;
        border-bottom: 1px solid var(--border-light);
    }

    .info-label {
        font-weight: 600;
        color: var(--text-primary);
    }

    .info-value {
        color: var(--text-secondary);
        text-align: right;
    }

    .full-description {
        background: var(--bg-secondary);
        padding: 1.5rem;
        border-radius: var(--radius-md);
        border: 1px solid var(--border-color);
        margin-bottom: 2rem;
    }

    .full-description h4 {
        margin-top: 0;
        margin-bottom: 1rem;
        color: var(--text-primary);
    }

    .description-content {
        line-height: 1.6;
        color: var(--text-secondary);
    }

    .comments-section {
        background: var(--bg-secondary);
        padding: 1.5rem;
        border-radius: var(--radius-md);
        border: 1px solid var(--border-color);
    }

    .comments-section h4 {
        margin-top: 0;
        margin-bottom: 1rem;
        color: var(--text-primary);
    }

    .comment-form {
        margin-bottom: 1.5rem;
    }

    .comment-input {
        width: 100%;
        padding: 0.75rem;
        border: 1px solid var(--border-color);
        border-radius: var(--radius-sm);
        margin-bottom: 0.5rem;
        resize: vertical;
        min-height: 80px;
    }

    .comments-list {
        max-height: 300px;
        overflow-y: auto;
    }

    .comment-item {
        background: var(--bg-primary);
        padding: 1rem;
        border-radius: var(--radius-sm);
        margin-bottom: 1rem;
        border-left: 4px solid var(--primary);
    }

    .comment-header {
        display: flex;
        justify-content: space-between;
        margin-bottom: 0.5rem;
    }

    .comment-author {
        font-weight: 600;
        color: var(--text-primary);
    }

    .comment-date {
        color: var(--text-muted);
        font-size: 0.8rem;
    }

    .comment-text {
        color: var(--text-secondary);
        line-height: 1.5;
    }

    .no-comments {
        text-align: center;
        color: var(--text-muted);
        padding: 2rem;
        font-style: italic;
    }

    @media (max-width: 768px) {
        .task-details-grid {
            grid-template-columns: 1fr;
            gap: 1rem;
        }
    }
    </style>

    <script>
    async function viewTaskDetails(taskId) {
        // Show loading state
        document.getElementById('taskModalBody').innerHTML = `
            <div class="loading-state">
                <div class="loading-spinner"></div>
                <p>Loading task details...</p>
            </div>
        `;
        
        // Show modal
        document.getElementById('taskModal').style.display = 'flex';
        
        try {
            // Fetch task details from server
            const response = await fetch(`tasks.php?action=get_task_details&task_id=${taskId}`);
            const data = await response.json();
            
            if (data.success) {
                displayTaskDetails(data.task, data.comments);
            } else {
                document.getElementById('taskModalBody').innerHTML = `
                    <div class="alert alert-error">
                        <div class="alert-content">
                            <span class="alert-icon">❌</span>
                            ${data.message}
                        </div>
                    </div>
                `;
            }
        } catch (error) {
            document.getElementById('taskModalBody').innerHTML = `
                <div class="alert alert-error">
                    <div class="alert-content">
                        <span class="alert-icon">❌</span>
                        Error loading task details: ${error.message}
                    </div>
                </div>
            `;
        }
    }

    function displayTaskDetails(task, comments) {
        const modalBody = document.getElementById('taskModalBody');
        
        modalBody.innerHTML = `
            <div class="task-details">
                <!-- Task Information Grid -->
                <div class="task-details-grid">
                    <div class="task-info-section">
                        <h4>📋 Task Information</h4>
                        <div class="info-item">
                            <span class="info-label">Title:</span>
                            <span class="info-value">${escapeHtml(task.title)}</span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Status:</span>
                            <span class="info-value">
                                <span class="badge badge-${getStatusBadgeClass(task.status)}">
                                    ${task.status.charAt(0).toUpperCase() + task.status.slice(1)}
                                </span>
                            </span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Priority:</span>
                            <span class="info-value">
                                <span class="badge badge-${getPriorityBadgeClass(task.priority)}">
                                    ${task.priority.charAt(0).toUpperCase() + task.priority.slice(1)}
                                </span>
                            </span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Due Date:</span>
                            <span class="info-value">
                                <span class="date-badge ${isOverdue(task.due_date) ? 'overdue' : ''}">
                                    ${formatDate(task.due_date)}
                                </span>
                            </span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Created:</span>
                            <span class="info-value">${formatDate(task.created_at)}</span>
                        </div>
                        ${task.completion_date ? `
                        <div class="info-item">
                            <span class="info-label">Completed:</span>
                            <span class="info-value">${formatDate(task.completion_date)}</span>
                        </div>
                        ` : ''}
                    </div>

                    <div class="task-assignment-section">
                        <h4>👥 Assignment Details</h4>
                        <div class="info-item">
                            <span class="info-label">Assigned To:</span>
                            <span class="info-value">${escapeHtml(task.assigned_to_name || 'Unassigned')}</span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Submitted By:</span>
                            <span class="info-value">${escapeHtml(task.submitted_by_name || 'Unknown')}</span>
                        </div>
                        ${task.assigned_by_name ? `
                        <div class="info-item">
                            <span class="info-label">Assigned By:</span>
                            <span class="info-value">${escapeHtml(task.assigned_by_name)}</span>
                        </div>
                        ` : ''}
                        <div class="info-item">
                            <span class="info-label">Approval:</span>
                            <span class="info-value">
                                ${task.is_approved ? 
                                    `<span class="badge badge-success">Approved</span>` : 
                                    `<span class="badge badge-warning">Pending Approval</span>`
                                }
                            </span>
                        </div>
                        ${task.approved_by_name ? `
                        <div class="info-item">
                            <span class="info-label">Approved By:</span>
                            <span class="info-value">${escapeHtml(task.approved_by_name)}</span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Approved At:</span>
                            <span class="info-value">${formatDate(task.approved_at)}</span>
                        </div>
                        ` : ''}
                    </div>
                </div>

                <!-- Full Description -->
                <div class="full-description">
                    <h4>📝 Description</h4>
                    <div class="description-content">
                        ${task.description ? escapeHtml(task.description).replace(/\n/g, '<br>') : '<em>No description provided</em>'}
                    </div>
                </div>

                <!-- Comments Section -->
                <div class="comments-section">
                    <h4>💬 Comments & Activity</h4>
                    
                    <!-- Add Comment Form -->
                    <form method="POST" class="comment-form" onsubmit="return addComment(${task.id}, this)">
                        <textarea name="comment" class="comment-input" placeholder="Add a comment..." required></textarea>
                        <input type="hidden" name="action" value="add_comment">
                        <input type="hidden" name="task_id" value="${task.id}">
                        <button type="submit" class="btn btn-primary btn-sm">Add Comment</button>
                    </form>

                    <!-- Comments List -->
                    <div class="comments-list">
                        ${comments.length > 0 ? 
                            comments.map(comment => `
                                <div class="comment-item">
                                    <div class="comment-header">
                                        <span class="comment-author">${escapeHtml(comment.user_name || 'Unknown User')}</span>
                                        <span class="comment-date">${formatDateTime(comment.created_at)}</span>
                                    </div>
                                    <div class="comment-text">${escapeHtml(comment.comment).replace(/\n/g, '<br>')}</div>
                                </div>
                            `).join('') : 
                            '<div class="no-comments">No comments yet. Be the first to comment!</div>'
                        }
                    </div>
                </div>
            </div>
        `;
    }

    function addComment(taskId, form) {
        const formData = new FormData(form);
        
        fetch('tasks.php', {
            method: 'POST',
            body: formData
        }).then(response => {
            // Refresh the page to show the new comment
            window.location.reload();
        });
        
        return false; // Prevent default form submission
    }

    function escapeHtml(unsafe) {
        if (!unsafe) return '';
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }

    function formatDate(dateString) {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', { 
            year: 'numeric', 
            month: 'short', 
            day: 'numeric' 
        });
    }

    function formatDateTime(dateString) {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', { 
            year: 'numeric', 
            month: 'short', 
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    function getStatusBadgeClass(status) {
        switch (status) {
            case 'completed': return 'success';
            case 'in-progress': return 'info';
            case 'blocked': return 'danger';
            case 'rejected': return 'danger';
            default: return 'warning';
        }
    }

    function getPriorityBadgeClass(priority) {
        switch (priority) {
            case 'high': return 'danger';
            case 'medium': return 'warning';
            default: return 'info';
        }
    }

    function isOverdue(dueDate) {
        if (!dueDate) return false;
        const today = new Date();
        const due = new Date(dueDate);
        return due < today;
    }

    function closeTaskModal() {
        document.getElementById('taskModal').style.display = 'none';
    }

    function filterTasks() {
        const statusFilter = document.getElementById('statusFilter').value;
        const priorityFilter = document.getElementById('priorityFilter').value;
        const approvalFilter = document.getElementById('approvalFilter').value;
        
        const rows = document.querySelectorAll('.task-row');
        let visibleCount = 0;
        
        rows.forEach(row => {
            const status = row.getAttribute('data-status');
            const priority = row.getAttribute('data-priority');
            const approved = row.getAttribute('data-approved');
            
            const statusMatch = !statusFilter || status === statusFilter;
            const priorityMatch = !priorityFilter || priority === priorityFilter;
            const approvalMatch = !approvalFilter || approved === approvalFilter;
            
            if (statusMatch && priorityMatch && approvalMatch) {
                row.classList.remove('hidden');
                visibleCount++;
            } else {
                row.classList.add('hidden');
            }
        });
        
        document.getElementById('visibleTasksCount').textContent = visibleCount;
    }

    function searchTasks() {
        const searchTerm = document.getElementById('searchFilter').value.toLowerCase();
        const rows = document.querySelectorAll('.task-row');
        let visibleCount = 0;
        
        rows.forEach(row => {
            const searchData = row.getAttribute('data-search');
            if (searchData.includes(searchTerm)) {
                row.classList.remove('hidden');
                visibleCount++;
            } else {
                row.classList.add('hidden');
            }
        });
        
        document.getElementById('visibleTasksCount').textContent = visibleCount;
    }

    function resetFilters() {
        document.getElementById('statusFilter').value = '';
        document.getElementById('priorityFilter').value = '';
        document.getElementById('approvalFilter').value = '';
        document.getElementById('searchFilter').value = '';
        filterTasks();
        searchTasks();
    }

    function refreshPage() {
        window.location.reload();
    }

    function exportTasks() {
        alert('Export functionality would be implemented here. This would generate a CSV/Excel file of all tasks.');
    }

    // Close modal when clicking outside
    document.getElementById('taskModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeTaskModal();
        }
    });

    // Auto-hide alerts
    setTimeout(() => {
        const alerts = document.querySelectorAll('.alert');
        alerts.forEach(alert => {
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.style.display = 'none';
            }, 500);
        });
    }, 5000);

    // Initialize page
    document.addEventListener('DOMContentLoaded', function() {
        console.log('Total tasks loaded:', <?php echo count($all_tasks); ?>);
    });
    </script>
</body>
</html>

<?php
function getStatusBadgeClass($status) {
    switch ($status) {
        case 'completed': return 'success';
        case 'in-progress': return 'info';
        case 'blocked': return 'danger';
        case 'rejected': return 'danger';
        default: return 'warning';
    }
}

function getPriorityBadgeClass($priority) {
    switch ($priority) {
        case 'high': return 'danger';
        case 'medium': return 'warning';
        default: return 'info';
    }
}

function isOverdue($due_date) {
    if (!$due_date) return false;
    return strtotime($due_date) < strtotime(date('Y-m-d'));
}
?>