<?php
include_once '../../includes/auth.php';
include_once '../../config/database.php';
include_once '../../models/User.php';
include_once '../../models/Task.php';

redirectIfNotLoggedIn();
if ($_SESSION['user_role'] !== 'admin') {
    header('Location: ../user/');
    exit;
}

$database = new Database();
$db = $database->getConnection();
$user = new User($db);
$task = new Task($db);

// Handle user actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'toggle_user_status':
            if (isset($_POST['user_id'])) {
                $user_id = $_POST['user_id'];
                $current_user = $user->getUserById($user_id);
                
                if ($current_user && $current_user['id'] != $_SESSION['user_id']) {
                    $new_status = $current_user['is_active'] ? 0 : 1;
                    if ($user->updateUserStatus($user_id, $new_status)) {
                        $_SESSION['success'] = 'User status updated successfully.';
                    } else {
                        $_SESSION['error'] = 'Failed to update user status.';
                    }
                } else {
                    $_SESSION['error'] = 'Cannot modify your own status.';
                }
            }
            break;
            
        case 'delete_user':
            if (isset($_POST['user_id'])) {
                $user_id = $_POST['user_id'];
                $target_user = $user->getUserById($user_id);
                
                if ($target_user && $target_user['role'] !== 'admin' && $target_user['id'] != $_SESSION['user_id']) {
                    if ($user->deleteUser($user_id)) {
                        $_SESSION['success'] = 'User deleted successfully.';
                    } else {
                        $_SESSION['error'] = 'Failed to delete user.';
                    }
                } else {
                    $_SESSION['error'] = 'Cannot delete admin users or your own account.';
                }
            }
            break;
            
        case 'create_user':
            $new_user = new User($db);
            $new_user->name = $_POST['name'];
            $new_user->email = $_POST['email'];
            $new_user->password = $_POST['password'];
            $new_user->role = $_POST['role'];
            
            if (!$user->emailExists($new_user->email)) {
                if ($new_user->create()) {
                    $_SESSION['success'] = 'User created successfully.';
                } else {
                    $_SESSION['error'] = 'Failed to create user.';
                }
            } else {
                $_SESSION['error'] = 'Email already exists.';
            }
            break;
    }
    
    header('Location: users.php');
    exit;
}

// Get all users
$users = $user->getAllUsers();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Users - Admin</title>
    <link rel="stylesheet" href="../../assets/css/style.css">
</head>
<body>
    <div class="app-container">
        <?php include '../../includes/sidebar.php'; ?>
        
        <div class="main-content">
            <div class="main-header">
                <div class="header-content">
                    <h1>Manage Users</h1>
                    <div class="user-menu">
                        <button type="button" class="btn btn-success" onclick="openCreateUserModal()">Create New User</button>
                    </div>
                </div>
            </div>

            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success"><?php echo $_SESSION['success']; unset($_SESSION['success']); ?></div>
            <?php endif; ?>

            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-error"><?php echo $_SESSION['error']; unset($_SESSION['error']); ?></div>
            <?php endif; ?>

            <!-- Users Table -->
            <div class="card">
                <div class="card-header">
                    <h2>All Users (<?php echo $users->rowCount(); ?>)</h2>
                </div>
                <div class="card-body">
                    <div class="table-container">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>Email</th>
                                    <th>Role</th>
                                    <th>Status</th>
                                    <th>Tasks Assigned</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($user_item = $users->fetch(PDO::FETCH_ASSOC)): 
                                    $task_count = $task->getUserTaskCount($user_item['id']);
                                ?>
                                <tr>
                                    <td>
                                        <strong><?php echo htmlspecialchars($user_item['name']); ?></strong>
                                        <?php if ($user_item['id'] == $_SESSION['user_id']): ?>
                                            <span class="badge badge-info">You</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($user_item['email']); ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $user_item['role'] === 'admin' ? 'danger' : 'info'; ?>">
                                            <?php echo ucfirst($user_item['role']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge badge-<?php echo $user_item['is_active'] ? 'success' : 'danger'; ?>">
                                            <?php echo $user_item['is_active'] ? 'Active' : 'Inactive'; ?>
                                        </span>
                                    </td>
                                    <td><?php echo $task_count; ?></td>
                                    <td><?php echo date('M j, Y', strtotime($user_item['created_at'])); ?></td>
                                    <td>
                                        <?php if ($user_item['id'] != $_SESSION['user_id']): ?>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="action" value="toggle_user_status">
                                                <input type="hidden" name="user_id" value="<?php echo $user_item['id']; ?>">
                                                <button type="submit" class="btn btn-<?php echo $user_item['is_active'] ? 'warning' : 'success'; ?>">
                                                    <?php echo $user_item['is_active'] ? 'Deactivate' : 'Activate'; ?>
                                                </button>
                                            </form>
                                            <?php if ($user_item['role'] !== 'admin'): ?>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="action" value="delete_user">
                                                    <input type="hidden" name="user_id" value="<?php echo $user_item['id']; ?>">
                                                    <button type="submit" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete this user?')">Delete</button>
                                                </form>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span class="badge badge-info">Current User</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Create User Modal -->
    <div id="createUserModal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="card">
                <div class="card-header">
                    <h2>Create New User</h2>
                    <button type="button" onclick="closeCreateUserModal()" class="btn btn-danger">&times;</button>
                </div>
                <div class="card-body">
                    <form method="POST" id="createUserForm">
                        <input type="hidden" name="action" value="create_user">
                        
                        <div class="form-group">
                            <label for="name">Full Name *</label>
                            <input type="text" id="name" name="name" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="email">Email Address *</label>
                            <input type="email" id="email" name="email" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="password">Password *</label>
                            <input type="password" id="password" name="password" class="form-control" required minlength="6">
                        </div>
                        
                        <div class="form-group">
                            <label for="role">Role *</label>
                            <select id="role" name="role" class="form-control" required>
                                <option value="user">User</option>
                                <option value="admin">Admin</option>
                            </select>
                        </div>
                        
                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">Create User</button>
                            <button type="button" onclick="closeCreateUserModal()" class="btn btn-secondary">Cancel</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <style>
        .modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
        }
        
        .modal-content {
            background: var(--bg-card);
            border-radius: var(--radius-lg);
            width: 90%;
            max-width: 500px;
            max-height: 90vh;
            overflow-y: auto;
        }
        
        .form-actions {
            display: flex;
            gap: 1rem;
            margin-top: 1.5rem;
        }
    </style>

    <script>
        function openCreateUserModal() {
            document.getElementById('createUserModal').style.display = 'flex';
        }
        
        function closeCreateUserModal() {
            document.getElementById('createUserModal').style.display = 'none';
            document.getElementById('createUserForm').reset();
        }
        
        // Close modal when clicking outside
        document.getElementById('createUserModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeCreateUserModal();
            }
        });
        
        setTimeout(() => {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                alert.style.display = 'none';
            });
        }, 5000);
    </script>
</body>
</html>