<?php
include_once '../../includes/auth.php';
include_once '../../config/database.php';
include_once '../../models/Task.php';

redirectIfNotLoggedIn();
checkSessionTimeout();

if ($_SESSION['user_role'] === 'admin') {
    header('Location: ../admin/');
    exit;
}

$database = new Database();
$db = $database->getConnection();
$task = new Task($db);

$user_id = $_SESSION['user_id'];

try {
    // Get user-specific statistics
    $my_tasks = $task->getUserTasks($user_id);
    $pending_tasks = $task->getUserTasksByStatus($user_id, 'pending');
    $in_progress_tasks = $task->getUserTasksByStatus($user_id, 'in-progress');
    $completed_tasks = $task->getUserTasksByStatus($user_id, 'completed');
    $recent_tasks = $task->getRecentUserTasks($user_id, 5);
    $submitted_tasks = $task->getUserSubmittedTasks($user_id);
    
    $total_tasks = $my_tasks->rowCount();
    $pending_count = $pending_tasks->rowCount();
    $in_progress_count = $in_progress_tasks->rowCount();
    $completed_count = $completed_tasks->rowCount();
    $submitted_count = $submitted_tasks->rowCount();
} catch (Exception $e) {
    error_log("User dashboard error: " . $e->getMessage());
    $_SESSION['error'] = 'Unable to load dashboard data. Please try again.';
    $total_tasks = $pending_count = $in_progress_count = $completed_count = $submitted_count = 0;
    $recent_tasks = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Dashboard - Task Management System</title>
    <link rel="stylesheet" href="../../assets/css/style.css">
</head>
<body>
    <div class="app-container">
        <?php include '../../includes/sidebar.php'; ?>
        
        <div class="main-content">
            <div class="main-header">
                <div class="header-content">
                    <h1>My Dashboard</h1>
                    <div class="header-actions">
                        <span class="welcome-text">Hello, <?php echo htmlspecialchars($_SESSION['user_name']); ?>! 👋</span>
                        <div class="header-buttons">
                            <a href="submit-task.php" class="btn btn-success">
                                <span class="btn-icon">➕</span>
                                Submit New Task
                            </a>
                        </div>
                    </div>
                </div>
            </div>

            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success">
                    <div class="alert-content">
                        <span class="alert-icon">✅</span>
                        <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-error">
                    <div class="alert-content">
                        <span class="alert-icon">❌</span>
                        <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Statistics -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">📋</div>
                    <div class="stat-value"><?php echo $total_tasks; ?></div>
                    <div class="stat-label">Total Tasks</div>
                    <div class="stat-trend">Assigned to you</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">⏳</div>
                    <div class="stat-value"><?php echo $pending_count; ?></div>
                    <div class="stat-label">Pending</div>
                    <div class="stat-trend">Waiting for action</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">🗓️</div>
                    <div class="stat-value"><?php echo $in_progress_count; ?></div>
                    <div class="stat-label">In Progress</div>
                    <div class="stat-trend">Currently working on</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon">✅</div>
                    <div class="stat-value"><?php echo $completed_count; ?></div>
                    <div class="stat-label">Completed</div>
                    <div class="stat-trend">Great work!</div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="card">
                <div class="card-header">
                    <h2>Quick Actions</h2>
                    <p class="card-subtitle">Manage your tasks efficiently</p>
                </div>
                <div class="card-body">
                    <div class="quick-actions-grid">
                        <a href="my-tasks.php" class="quick-action-card">
                            <div class="action-icon">✅</div>
                            <div class="action-content">
                                <h3>View My Tasks</h3>
                                <p>See all tasks assigned to you and update their status</p>
                            </div>
                            <div class="action-badge"><?php echo $total_tasks; ?> tasks</div>
                        </a>
                        
                        <a href="submit-task.php" class="quick-action-card">
                            <div class="action-icon">➕</div>
                            <div class="action-content">
                                <h3>Submit New Task</h3>
                                <p>Request new tasks for yourself or team members</p>
                            </div>
                        </a>
                        
                        <a href="my-tasks.php?filter=pending" class="quick-action-card">
                            <div class="action-icon">⏳</div>
                            <div class="action-content">
                                <h3>Pending Tasks</h3>
                                <p>View tasks that need your attention</p>
                            </div>
                            <div class="action-badge"><?php echo $pending_count; ?> pending</div>
                        </a>
                        
                        <a href="my-tasks.php?filter=in-progress" class="quick-action-card">
                            <div class="action-icon">🗓️</div>
                            <div class="action-content">
                                <h3>In Progress</h3>
                                <p>Continue working on active tasks</p>
                            </div>
                            <div class="action-badge"><?php echo $in_progress_count; ?> active</div>
                        </a>
                    </div>
                </div>
            </div>

            <div class="grid-2-col">
                <!-- Recent Tasks -->
                <div class="card">
                    <div class="card-header">
                        <h2>Recent Tasks</h2>
                        <a href="my-tasks.php" class="btn btn-secondary">View All</a>
                    </div>
                    <div class="card-body">
                        <?php if ($recent_tasks && $recent_tasks->rowCount() > 0): ?>
                            <div class="task-list">
                                <?php while ($task_item = $recent_tasks->fetch(PDO::FETCH_ASSOC)): ?>
                                <div class="task-list-item">
                                    <div class="task-info">
                                        <h4><?php echo htmlspecialchars($task_item['title']); ?></h4>
                                        <p class="task-meta">
                                            Due: <?php echo date('M j, Y', strtotime($task_item['due_date'])); ?>
                                        </p>
                                    </div>
                                    <div class="task-status">
                                        <span class="badge badge-<?php echo getStatusBadgeClass($task_item['status']); ?>">
                                            <?php echo ucfirst($task_item['status']); ?>
                                        </span>
                                        <span class="task-priority badge badge-<?php echo getPriorityBadgeClass($task_item['priority']); ?>">
                                            <?php echo ucfirst($task_item['priority']); ?>
                                        </span>
                                    </div>
                                </div>
                                <?php endwhile; ?>
                            </div>
                        <?php else: ?>
                            <div class="empty-state">
                                <div class="empty-icon">📋</div>
                                <h3>No Tasks Assigned</h3>
                                <p>You don't have any tasks assigned yet.</p>
                                <a href="submit-task.php" class="btn btn-primary">Submit Your First Task</a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Task Progress -->
                <div class="card">
                    <div class="card-header">
                        <h2>Task Progress</h2>
                    </div>
                    <div class="card-body">
                        <div class="progress-stats">
                            <div class="progress-item">
                                <div class="progress-label">
                                    <span>Task Completion</span>
                                    <span><?php echo $total_tasks > 0 ? round(($completed_count / $total_tasks) * 100, 1) : 0; ?>%</span>
                                </div>
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: <?php echo $total_tasks > 0 ? ($completed_count / $total_tasks * 100) : 0; ?>%"></div>
                                </div>
                            </div>
                            
                            <div class="progress-item">
                                <div class="progress-label">
                                    <span>In Progress</span>
                                    <span><?php echo $in_progress_count; ?> tasks</span>
                                </div>
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: <?php echo $total_tasks > 0 ? ($in_progress_count / $total_tasks * 100) : 0; ?>%"></div>
                                </div>
                            </div>
                            
                            <div class="progress-item">
                                <div class="progress-label">
                                    <span>Pending Review</span>
                                    <span><?php echo $pending_count; ?> tasks</span>
                                </div>
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: <?php echo $total_tasks > 0 ? ($pending_count / $total_tasks * 100) : 0; ?>%"></div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="status-summary">
                            <h4>Tasks by Priority</h4>
                            <div class="priority-stats">
                                <?php
                                $high_priority = $medium_priority = $low_priority = 0;
                                if ($my_tasks) {
                                    $my_tasks->execute();
                                    while ($task_item = $my_tasks->fetch(PDO::FETCH_ASSOC)) {
                                        switch ($task_item['priority']) {
                                            case 'high': $high_priority++; break;
                                            case 'medium': $medium_priority++; break;
                                            case 'low': $low_priority++; break;
                                        }
                                    }
                                }
                                ?>
                                <div class="priority-stat">
                                    <span class="priority-dot high"></span>
                                    <span>High: <?php echo $high_priority; ?></span>
                                </div>
                                <div class="priority-stat">
                                    <span class="priority-dot medium"></span>
                                    <span>Medium: <?php echo $medium_priority; ?></span>
                                </div>
                                <div class="priority-stat">
                                    <span class="priority-dot low"></span>
                                    <span>Low: <?php echo $low_priority; ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Auto-hide alerts after 5 seconds
        setTimeout(() => {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                alert.style.opacity = '0';
                alert.style.transition = 'opacity 0.5s ease';
                setTimeout(() => {
                    alert.style.display = 'none';
                }, 500);
            });
        }, 5000);

        // Add loading states to buttons
        document.querySelectorAll('a.btn').forEach(button => {
            button.addEventListener('click', function(e) {
                const originalText = this.innerHTML;
                this.innerHTML = '<span class="loading-spinner"></span> Loading...';
                this.style.pointerEvents = 'none';
                
                setTimeout(() => {
                    this.innerHTML = originalText;
                    this.style.pointerEvents = 'auto';
                }, 1500);
            });
        });
    </script>
</body>
</html>

<?php
function getStatusBadgeClass($status) {
    switch ($status) {
        case 'completed': return 'success';
        case 'in-progress': return 'info';
        case 'blocked': return 'danger';
        default: return 'warning';
    }
}

function getPriorityBadgeClass($priority) {
    switch ($priority) {
        case 'high': return 'danger';
        case 'medium': return 'warning';
        default: return 'info';
    }
}
?>