<?php
include_once '../../includes/auth.php';
include_once '../../config/database.php';
include_once '../../models/Task.php';
include_once '../../models/User.php';

redirectIfNotLoggedIn();
checkSessionTimeout();

if ($_SESSION['user_role'] === 'admin') {
    header('Location: ../admin/');
    exit;
}

$database = new Database();
$db = $database->getConnection();
$task = new Task($db);
$user = new User($db);

$user_id = $_SESSION['user_id'];

// Handle AJAX request for task details
if (isset($_GET['action']) && $_GET['action'] === 'get_task_details') {
    header('Content-Type: application/json');
    
    try {
        $task_id = $_GET['task_id'] ?? 0;
        $task_details = $task->getTask($task_id);
        
        if ($task_details) {
            // Get task comments
            $comments_result = $task->getComments($task_id);
            $comments = $comments_result ? $comments_result->fetchAll(PDO::FETCH_ASSOC) : [];
            
            echo json_encode([
                'success' => true,
                'task' => $task_details,
                'comments' => $comments
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Task not found'
            ]);
        }
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => 'Error loading task details: ' . $e->getMessage()
        ]);
    }
    exit;
}

// Handle status update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        switch ($action) {
            case 'update_status':
                $task_id = $_POST['task_id'];
                $status = $_POST['status'];
                $comment = $_POST['comment'] ?? '';
                
                if ($task->updateStatus($task_id, $status, $comment, $user_id)) {
                    $_SESSION['success'] = 'Task status updated successfully.';
                } else {
                    $_SESSION['error'] = 'Failed to update task status.';
                }
                break;
                
            case 'add_comment':
                if (isset($_POST['task_id']) && isset($_POST['comment'])) {
                    if ($task->addComment($_POST['task_id'], $user_id, $_POST['comment'])) {
                        $_SESSION['success'] = 'Comment added successfully.';
                    } else {
                        $_SESSION['error'] = 'Failed to add comment.';
                    }
                }
                break;
        }
    } catch (Exception $e) {
        $_SESSION['error'] = $e->getMessage();
    }
    
    header('Location: my-tasks.php');
    exit;
}

// Get filter if any
$filter = $_GET['filter'] ?? 'all';

try {
    // Get user tasks based on filter
    switch ($filter) {
        case 'pending':
            $user_tasks = $task->getUserTasksByStatus($user_id, 'pending');
            $page_title = 'Pending Tasks';
            break;
        case 'in-progress':
            $user_tasks = $task->getUserTasksByStatus($user_id, 'in-progress');
            $page_title = 'Tasks In Progress';
            break;
        case 'completed':
            $user_tasks = $task->getUserTasksByStatus($user_id, 'completed');
            $page_title = 'Completed Tasks';
            break;
        case 'blocked':
            $user_tasks = $task->getUserTasksByStatus($user_id, 'blocked');
            $page_title = 'Blocked Tasks';
            break;
        default:
            $user_tasks = $task->getUserTasks($user_id);
            $page_title = 'All My Tasks';
            break;
    }
    
    $tasks_data = $user_tasks ? $user_tasks->fetchAll(PDO::FETCH_ASSOC) : [];
    $total_tasks = count($tasks_data);
} catch (Exception $e) {
    error_log("My tasks page error: " . $e->getMessage());
    $_SESSION['error'] = 'Unable to load tasks. Please try again.';
    $tasks_data = [];
    $total_tasks = 0;
    $page_title = 'My Tasks';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - Task Management System</title>
    <link rel="stylesheet" href="../../assets/css/style.css">
</head>
<body>
    <div class="app-container">
        <?php include '../../includes/sidebar.php'; ?>
        
        <div class="main-content">
            <div class="main-header">
                <div class="header-content">
                    <h1><?php echo $page_title; ?></h1>
                    <div class="header-actions">
                        <a href="submit-task.php" class="btn btn-success">
                            <span class="btn-icon">➕</span>
                            Submit New Task
                        </a>
                    </div>
                </div>
            </div>

            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success">
                    <div class="alert-content">
                        <span class="alert-icon">✅</span>
                        <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-error">
                    <div class="alert-content">
                        <span class="alert-icon">❌</span>
                        <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Tasks Filter -->
            <div class="card">
                <div class="card-body">
                    <div class="filter-tabs">
                        <a href="?filter=all" class="filter-tab <?php echo $filter === 'all' ? 'active' : ''; ?>">
                            All Tasks
                            <span class="filter-count"><?php echo $filter === 'all' ? $total_tasks : getTaskCountByFilter($task, $user_id, 'all'); ?></span>
                        </a>
                        <a href="?filter=pending" class="filter-tab <?php echo $filter === 'pending' ? 'active' : ''; ?>">
                            Pending
                            <span class="filter-count"><?php echo $filter === 'pending' ? $total_tasks : getTaskCountByFilter($task, $user_id, 'pending'); ?></span>
                        </a>
                        <a href="?filter=in-progress" class="filter-tab <?php echo $filter === 'in-progress' ? 'active' : ''; ?>">
                            In Progress
                            <span class="filter-count"><?php echo $filter === 'in-progress' ? $total_tasks : getTaskCountByFilter($task, $user_id, 'in-progress'); ?></span>
                        </a>
                        <a href="?filter=completed" class="filter-tab <?php echo $filter === 'completed' ? 'active' : ''; ?>">
                            Completed
                            <span class="filter-count"><?php echo $filter === 'completed' ? $total_tasks : getTaskCountByFilter($task, $user_id, 'completed'); ?></span>
                        </a>
                        <a href="?filter=blocked" class="filter-tab <?php echo $filter === 'blocked' ? 'active' : ''; ?>">
                            Blocked
                            <span class="filter-count"><?php echo $filter === 'blocked' ? $total_tasks : getTaskCountByFilter($task, $user_id, 'blocked'); ?></span>
                        </a>
                    </div>
                </div>
            </div>

            <!-- Tasks List -->
            <div class="card">
                <div class="card-header">
                    <h2><?php echo $page_title; ?> (<?php echo $total_tasks; ?>)</h2>
                    <div class="card-actions">
                        <button class="btn btn-secondary" onclick="toggleViewMode()">
                            <span id="viewModeIcon">📋</span>
                            <span id="viewModeText">Card View</span>
                        </button>
                    </div>
                </div>
                <div class="card-body">
                    <?php if ($total_tasks > 0): ?>
                        <div id="tableView" class="view-mode">
                            <div class="table-responsive">
                                <table class="data-table">
                                    <thead>
                                        <tr>
                                            <th>Task Details</th>
                                            <th>Status</th>
                                            <th>Due Date</th>
                                            <th>Priority</th>
                                            <th>Assigned By</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($tasks_data as $task_item): ?>
                                        <tr>
                                            <td>
                                                <div class="task-title"><?php echo htmlspecialchars($task_item['title']); ?></div>
                                                <div class="task-description"><?php echo htmlspecialchars(substr($task_item['description'] ?? '', 0, 100)); ?>...</div>
                                            </td>
                                            <td>
                                                <span class="badge badge-<?php echo getStatusBadgeClass($task_item['status']); ?>">
                                                    <?php echo ucfirst($task_item['status']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="date-badge <?php echo isOverdue($task_item['due_date']) ? 'overdue' : ''; ?>">
                                                    <?php echo date('M j, Y', strtotime($task_item['due_date'])); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge badge-<?php echo getPriorityBadgeClass($task_item['priority']); ?>">
                                                    <?php echo ucfirst($task_item['priority']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <div class="user-info-small">
                                                    <div class="user-name"><?php echo htmlspecialchars($task_item['assigned_by_name'] ?? 'Unknown'); ?></div>
                                                    <div class="user-date"><?php echo date('M j', strtotime($task_item['created_at'])); ?></div>
                                                </div>
                                            </td>
                                            <td>
                                                <div class="action-buttons">
                                                    <button type="button" class="btn btn-primary btn-sm" 
                                                            onclick="openStatusModal(<?php echo $task_item['id']; ?>, '<?php echo $task_item['status']; ?>')"
                                                            title="Update Status">
                                                        📝 Update
                                                    </button>
                                                    <button type="button" class="btn btn-secondary btn-sm" 
                                                            onclick="viewTaskDetails(<?php echo $task_item['id']; ?>)"
                                                            title="View Details">
                                                        👁️ View
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>

                        <div id="cardView" class="view-mode" style="display: none;">
                            <div class="tasks-grid">
                                <?php foreach ($tasks_data as $task_item): ?>
                                <div class="task-card <?php echo $task_item['status']; ?>">
                                    <div class="task-card-header">
                                        <h3><?php echo htmlspecialchars($task_item['title']); ?></h3>
                                        <div class="task-actions">
                                            <span class="priority-badge <?php echo $task_item['priority']; ?>">
                                                <?php echo ucfirst($task_item['priority']); ?>
                                            </span>
                                        </div>
                                    </div>
                                    <div class="task-card-body">
                                        <p class="task-description"><?php echo htmlspecialchars(substr($task_item['description'] ?? '', 0, 150)); ?>...</p>
                                        <div class="task-meta">
                                            <div class="meta-item">
                                                <strong>Status:</strong>
                                                <span class="status-badge <?php echo $task_item['status']; ?>">
                                                    <?php echo ucfirst($task_item['status']); ?>
                                                </span>
                                            </div>
                                            <div class="meta-item">
                                                <strong>Due Date:</strong>
                                                <span class="<?php echo isOverdue($task_item['due_date']) ? 'text-danger' : ''; ?>">
                                                    <?php echo date('M j, Y', strtotime($task_item['due_date'])); ?>
                                                </span>
                                            </div>
                                            <div class="meta-item">
                                                <strong>Assigned by:</strong>
                                                <?php echo htmlspecialchars($task_item['assigned_by_name'] ?? 'Unknown'); ?>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="task-card-footer">
                                        <button type="button" class="btn btn-primary btn-sm" 
                                                onclick="openStatusModal(<?php echo $task_item['id']; ?>, '<?php echo $task_item['status']; ?>')">
                                            Update Status
                                        </button>
                                        <button type="button" class="btn btn-secondary btn-sm" 
                                                onclick="viewTaskDetails(<?php echo $task_item['id']; ?>)">
                                            View Details
                                        </button>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php else: ?>
                        <div class="empty-state">
                            <div class="empty-icon">
                                <?php 
                                switch ($filter) {
                                    case 'pending': echo '⏳'; break;
                                    case 'in-progress': echo '🗓️'; break;
                                    case 'completed': echo '✅'; break;
                                    case 'blocked': echo '🚧'; break;
                                    default: echo '📋'; break;
                                }
                                ?>
                            </div>
                            <h3>No Tasks Found</h3>
                            <p>
                                <?php 
                                switch ($filter) {
                                    case 'pending': 
                                        echo "You don't have any pending tasks at the moment."; 
                                        break;
                                    case 'in-progress': 
                                        echo "You don't have any tasks in progress right now."; 
                                        break;
                                    case 'completed': 
                                        echo "You haven't completed any tasks yet."; 
                                        break;
                                    case 'blocked': 
                                        echo "You don't have any blocked tasks."; 
                                        break;
                                    default: 
                                        echo "You don't have any tasks assigned yet."; 
                                        break;
                                }
                                ?>
                            </p>
                            <a href="submit-task.php" class="btn btn-primary">Submit Your First Task</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Status Update Modal -->
    <div id="statusModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Update Task Status</h3>
                <button type="button" class="close-btn" onclick="closeStatusModal()">&times;</button>
            </div>
            <div class="modal-body">
                <form id="statusForm" method="POST">
                    <input type="hidden" name="action" value="update_status">
                    <input type="hidden" name="task_id" id="task_id">
                    
                    <div class="form-group">
                        <label for="status">Status:</label>
                        <select name="status" id="status" class="form-control" onchange="toggleCommentField()" required>
                            <option value="pending">⏳ Pending</option>
                            <option value="in-progress">🗓️ In Progress</option>
                            <option value="completed">✅ Completed</option>
                            <option value="blocked">🚧 Blocked</option>
                        </select>
                    </div>
                    
                    <div class="form-group" id="commentField" style="display: none;">
                        <label for="comment">Progress Comment:</label>
                        <textarea name="comment" id="comment" class="form-control" 
                                  placeholder="Describe your progress, any issues, or completion details..." 
                                  rows="4"></textarea>
                        <small class="form-text">Adding a comment helps track your progress and any challenges.</small>
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary">
                            <span class="btn-icon">💾</span>
                            Update Status
                        </button>
                        <button type="button" onclick="closeStatusModal()" class="btn btn-secondary">Cancel</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Task Details Modal -->
    <div id="taskModal" class="modal">
        <div class="modal-content large">
            <div class="modal-header">
                <h3>Task Details</h3>
                <button type="button" class="close-btn" onclick="closeTaskModal()">&times;</button>
            </div>
            <div class="modal-body" id="taskModalBody">
                <!-- Content will be loaded dynamically -->
            </div>
        </div>
    </div>

    <style>
    .filter-tabs {
        display: flex;
        gap: 0.5rem;
        flex-wrap: wrap;
    }

    .filter-tab {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.75rem 1.25rem;
        background: var(--bg-tertiary);
        color: var(--text-secondary);
        text-decoration: none;
        border-radius: var(--radius);
        transition: all 0.3s ease;
        border: 1px solid transparent;
    }

    .filter-tab:hover {
        background: var(--bg-secondary);
        color: var(--text-primary);
    }

    .filter-tab.active {
        background: var(--primary);
        color: white;
        border-color: var(--primary);
    }

    .filter-count {
        background: var(--bg-primary);
        color: var(--text-primary);
        padding: 0.2rem 0.5rem;
        border-radius: 10px;
        font-size: 0.8rem;
        font-weight: 600;
    }

    .filter-tab.active .filter-count {
        background: rgba(255, 255, 255, 0.2);
        color: white;
    }

    .view-mode {
        transition: all 0.3s ease;
    }

    .tasks-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
        gap: 1.5rem;
    }

    .task-card {
        background: var(--bg-card);
        border: 1px solid var(--border);
        border-radius: var(--radius-lg);
        overflow: hidden;
        transition: all 0.3s ease;
    }

    .task-card:hover {
        transform: translateY(-2px);
        box-shadow: var(--shadow-lg);
    }

    .task-card-header {
        padding: 1.25rem;
        border-bottom: 1px solid var(--border);
        display: flex;
        justify-content: space-between;
        align-items: flex-start;
        gap: 1rem;
    }

    .task-card-header h3 {
        margin: 0;
        color: var(--text-primary);
        font-size: 1.1rem;
        line-height: 1.4;
    }

    .task-card-body {
        padding: 1.25rem;
    }

    .task-card-footer {
        padding: 1.25rem;
        border-top: 1px solid var(--border);
        display: flex;
        gap: 0.75rem;
    }

    .priority-badge {
        padding: 0.3rem 0.75rem;
        border-radius: 15px;
        font-size: 0.8rem;
        font-weight: 600;
        text-transform: uppercase;
    }

    .priority-badge.high {
        background: rgba(239, 68, 68, 0.15);
        color: var(--danger);
        border: 1px solid rgba(239, 68, 68, 0.3);
    }

    .priority-badge.medium {
        background: rgba(245, 158, 11, 0.15);
        color: var(--warning);
        border: 1px solid rgba(245, 158, 11, 0.3);
    }

    .priority-badge.low {
        background: rgba(16, 185, 129, 0.15);
        color: var(--success);
        border: 1px solid rgba(16, 185, 129, 0.3);
    }

    .card-actions {
        display: flex;
        gap: 0.75rem;
        align-items: center;
    }

    .text-danger {
        color: var(--danger) !important;
        font-weight: 600;
    }

    /* Task Details Styles */
    .task-details-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 2rem;
        margin-bottom: 2rem;
    }

    .task-info-section, .task-assignment-section {
        background: var(--bg-secondary);
        padding: 1.5rem;
        border-radius: var(--radius-md);
        border: 1px solid var(--border-color);
    }

    .task-info-section h4, .task-assignment-section h4 {
        margin-top: 0;
        margin-bottom: 1rem;
        color: var(--text-primary);
        border-bottom: 2px solid var(--primary);
        padding-bottom: 0.5rem;
    }

    .info-item {
        display: flex;
        justify-content: space-between;
        margin-bottom: 0.75rem;
        padding-bottom: 0.75rem;
        border-bottom: 1px solid var(--border-light);
    }

    .info-label {
        font-weight: 600;
        color: var(--text-primary);
    }

    .info-value {
        color: var(--text-secondary);
        text-align: right;
    }

    .full-description {
        background: var(--bg-secondary);
        padding: 1.5rem;
        border-radius: var(--radius-md);
        border: 1px solid var(--border-color);
        margin-bottom: 2rem;
    }

    .full-description h4 {
        margin-top: 0;
        margin-bottom: 1rem;
        color: var(--text-primary);
    }

    .description-content {
        line-height: 1.6;
        color: var(--text-secondary);
        white-space: pre-wrap;
    }

    .comments-section {
        background: var(--bg-secondary);
        padding: 1.5rem;
        border-radius: var(--radius-md);
        border: 1px solid var(--border-color);
    }

    .comments-section h4 {
        margin-top: 0;
        margin-bottom: 1rem;
        color: var(--text-primary);
    }

    .comment-form {
        margin-bottom: 1.5rem;
    }

    .comment-input {
        width: 100%;
        padding: 0.75rem;
        border: 1px solid var(--border-color);
        border-radius: var(--radius-sm);
        margin-bottom: 0.5rem;
        resize: vertical;
        min-height: 80px;
    }

    .comments-list {
        max-height: 300px;
        overflow-y: auto;
    }

    .comment-item {
        background: var(--bg-primary);
        padding: 1rem;
        border-radius: var(--radius-sm);
        margin-bottom: 1rem;
        border-left: 4px solid var(--primary);
    }

    .comment-header {
        display: flex;
        justify-content: space-between;
        margin-bottom: 0.5rem;
    }

    .comment-author {
        font-weight: 600;
        color: var(--text-primary);
    }

    .comment-date {
        color: var(--text-muted);
        font-size: 0.8rem;
    }

    .comment-text {
        color: var(--text-secondary);
        line-height: 1.5;
        white-space: pre-wrap;
    }

    .no-comments {
        text-align: center;
        color: var(--text-muted);
        padding: 2rem;
        font-style: italic;
    }

    @media (max-width: 768px) {
        .task-details-grid {
            grid-template-columns: 1fr;
            gap: 1rem;
        }
    }
    </style>

    <script>
    let currentViewMode = 'table';

    function toggleViewMode() {
        const tableView = document.getElementById('tableView');
        const cardView = document.getElementById('cardView');
        const viewModeIcon = document.getElementById('viewModeIcon');
        const viewModeText = document.getElementById('viewModeText');

        if (currentViewMode === 'table') {
            tableView.style.display = 'none';
            cardView.style.display = 'block';
            viewModeIcon.textContent = '📊';
            viewModeText.textContent = 'Table View';
            currentViewMode = 'card';
        } else {
            tableView.style.display = 'block';
            cardView.style.display = 'none';
            viewModeIcon.textContent = '📋';
            viewModeText.textContent = 'Card View';
            currentViewMode = 'table';
        }
    }

    function openStatusModal(taskId, currentStatus) {
        document.getElementById('task_id').value = taskId;
        document.getElementById('status').value = currentStatus;
        document.getElementById('statusModal').style.display = 'flex';
        toggleCommentField();
    }
    
    function closeStatusModal() {
        document.getElementById('statusModal').style.display = 'none';
        document.getElementById('statusForm').reset();
    }
    
    function toggleCommentField() {
        const status = document.getElementById('status').value;
        const commentField = document.getElementById('commentField');
        commentField.style.display = (status !== 'pending') ? 'block' : 'none';
    }

    async function viewTaskDetails(taskId) {
        // Show loading state
        document.getElementById('taskModalBody').innerHTML = `
            <div class="loading-state">
                <div class="loading-spinner"></div>
                <p>Loading task details...</p>
            </div>
        `;
        
        // Show modal
        document.getElementById('taskModal').style.display = 'flex';
        
        try {
            // Fetch task details from server
            const response = await fetch(`my-tasks.php?action=get_task_details&task_id=${taskId}`);
            const data = await response.json();
            
            if (data.success) {
                displayTaskDetails(data.task, data.comments);
            } else {
                document.getElementById('taskModalBody').innerHTML = `
                    <div class="alert alert-error">
                        <div class="alert-content">
                            <span class="alert-icon">❌</span>
                            ${data.message}
                        </div>
                    </div>
                `;
            }
        } catch (error) {
            document.getElementById('taskModalBody').innerHTML = `
                <div class="alert alert-error">
                    <div class="alert-content">
                        <span class="alert-icon">❌</span>
                        Error loading task details: ${error.message}
                    </div>
                </div>
            `;
        }
    }

    function displayTaskDetails(task, comments) {
        const modalBody = document.getElementById('taskModalBody');
        
        modalBody.innerHTML = `
            <div class="task-details">
                <!-- Task Information Grid -->
                <div class="task-details-grid">
                    <div class="task-info-section">
                        <h4>📋 Task Information</h4>
                        <div class="info-item">
                            <span class="info-label">Title:</span>
                            <span class="info-value">${escapeHtml(task.title)}</span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Status:</span>
                            <span class="info-value">
                                <span class="badge badge-${getStatusBadgeClass(task.status)}">
                                    ${task.status.charAt(0).toUpperCase() + task.status.slice(1)}
                                </span>
                            </span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Priority:</span>
                            <span class="info-value">
                                <span class="badge badge-${getPriorityBadgeClass(task.priority)}">
                                    ${task.priority.charAt(0).toUpperCase() + task.priority.slice(1)}
                                </span>
                            </span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Due Date:</span>
                            <span class="info-value">
                                <span class="date-badge ${isOverdue(task.due_date) ? 'overdue' : ''}">
                                    ${formatDate(task.due_date)}
                                </span>
                            </span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Created:</span>
                            <span class="info-value">${formatDate(task.created_at)}</span>
                        </div>
                        ${task.completion_date ? `
                        <div class="info-item">
                            <span class="info-label">Completed:</span>
                            <span class="info-value">${formatDate(task.completion_date)}</span>
                        </div>
                        ` : ''}
                    </div>

                    <div class="task-assignment-section">
                        <h4>👥 Assignment Details</h4>
                        <div class="info-item">
                            <span class="info-label">Assigned To:</span>
                            <span class="info-value">${escapeHtml(task.assigned_to_name || 'Unassigned')}</span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Submitted By:</span>
                            <span class="info-value">${escapeHtml(task.submitted_by_name || 'Unknown')}</span>
                        </div>
                        ${task.assigned_by_name ? `
                        <div class="info-item">
                            <span class="info-label">Assigned By:</span>
                            <span class="info-value">${escapeHtml(task.assigned_by_name)}</span>
                        </div>
                        ` : ''}
                        <div class="info-item">
                            <span class="info-label">Approval:</span>
                            <span class="info-value">
                                ${task.is_approved ? 
                                    `<span class="badge badge-success">Approved</span>` : 
                                    `<span class="badge badge-warning">Pending Approval</span>`
                                }
                            </span>
                        </div>
                        ${task.approved_by_name ? `
                        <div class="info-item">
                            <span class="info-label">Approved By:</span>
                            <span class="info-value">${escapeHtml(task.approved_by_name)}</span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Approved At:</span>
                            <span class="info-value">${formatDate(task.approved_at)}</span>
                        </div>
                        ` : ''}
                    </div>
                </div>

                <!-- Full Description -->
                <div class="full-description">
                    <h4>📝 Description</h4>
                    <div class="description-content">
                        ${task.description ? escapeHtml(task.description) : '<em>No description provided</em>'}
                    </div>
                </div>

                <!-- Comments Section -->
                <div class="comments-section">
                    <h4>💬 Comments & Activity</h4>
                    
                    <!-- Add Comment Form -->
                    <form method="POST" class="comment-form" onsubmit="return addComment(${task.id}, this)">
                        <textarea name="comment" class="comment-input" placeholder="Add a comment..." required></textarea>
                        <input type="hidden" name="action" value="add_comment">
                        <input type="hidden" name="task_id" value="${task.id}">
                        <button type="submit" class="btn btn-primary btn-sm">Add Comment</button>
                    </form>

                    <!-- Comments List -->
                    <div class="comments-list">
                        ${comments.length > 0 ? 
                            comments.map(comment => `
                                <div class="comment-item">
                                    <div class="comment-header">
                                        <span class="comment-author">${escapeHtml(comment.user_name || 'Unknown User')}</span>
                                        <span class="comment-date">${formatDateTime(comment.created_at)}</span>
                                    </div>
                                    <div class="comment-text">${escapeHtml(comment.comment)}</div>
                                </div>
                            `).join('') : 
                            '<div class="no-comments">No comments yet. Be the first to comment!</div>'
                        }
                    </div>
                </div>
            </div>
        `;
    }

    function addComment(taskId, form) {
        const formData = new FormData(form);
        
        fetch('my-tasks.php', {
            method: 'POST',
            body: formData
        }).then(response => {
            // Refresh the page to show the new comment
            window.location.reload();
        });
        
        return false; // Prevent default form submission
    }

    function escapeHtml(unsafe) {
        if (!unsafe) return '';
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }

    function formatDate(dateString) {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', { 
            year: 'numeric', 
            month: 'short', 
            day: 'numeric' 
        });
    }

    function formatDateTime(dateString) {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        return date.toLocaleDateString('en-US', { 
            year: 'numeric', 
            month: 'short', 
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    function getStatusBadgeClass(status) {
        switch (status) {
            case 'completed': return 'success';
            case 'in-progress': return 'info';
            case 'blocked': return 'danger';
            case 'rejected': return 'danger';
            default: return 'warning';
        }
    }

    function getPriorityBadgeClass(priority) {
        switch (priority) {
            case 'high': return 'danger';
            case 'medium': return 'warning';
            default: return 'info';
        }
    }

    function isOverdue(dueDate) {
        if (!dueDate) return false;
        const today = new Date();
        const due = new Date(dueDate);
        return due < today;
    }

    function closeTaskModal() {
        document.getElementById('taskModal').style.display = 'none';
    }

    // Close modals when clicking outside
    document.querySelectorAll('.modal').forEach(modal => {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                if (this.id === 'statusModal') closeStatusModal();
                if (this.id === 'taskModal') closeTaskModal();
            }
        });
    });

    // Auto-hide alerts
    setTimeout(() => {
        const alerts = document.querySelectorAll('.alert');
        alerts.forEach(alert => {
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.style.display = 'none';
            }, 500);
        });
    }, 5000);

    // Form submission handling
    document.getElementById('statusForm').addEventListener('submit', function(e) {
        const submitBtn = this.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<span class="loading-spinner"></span> Updating...';
        submitBtn.disabled = true;
        
        setTimeout(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }, 2000);
    });
    </script>
</body>
</html>

<?php
function getStatusBadgeClass($status) {
    switch ($status) {
        case 'completed': return 'success';
        case 'in-progress': return 'info';
        case 'blocked': return 'danger';
        default: return 'warning';
    }
}

function getPriorityBadgeClass($priority) {
    switch ($priority) {
        case 'high': return 'danger';
        case 'medium': return 'warning';
        default: return 'info';
    }
}

function isOverdue($due_date) {
    if (!$due_date) return false;
    return strtotime($due_date) < strtotime(date('Y-m-d'));
}

function getTaskCountByFilter($task, $user_id, $filter) {
    try {
        switch ($filter) {
            case 'pending':
                $result = $task->getUserTasksByStatus($user_id, 'pending');
                break;
            case 'in-progress':
                $result = $task->getUserTasksByStatus($user_id, 'in-progress');
                break;
            case 'completed':
                $result = $task->getUserTasksByStatus($user_id, 'completed');
                break;
            case 'blocked':
                $result = $task->getUserTasksByStatus($user_id, 'blocked');
                break;
            default:
                $result = $task->getUserTasks($user_id);
                break;
        }
        $data = $result ? $result->fetchAll(PDO::FETCH_ASSOC) : [];
        return count($data);
    } catch (Exception $e) {
        error_log("Error getting task count: " . $e->getMessage());
        return 0;
    }
}