<?php
include_once '../../includes/auth.php';
include_once '../../config/database.php';
include_once '../../models/Task.php';
include_once '../../models/User.php';

redirectIfNotLoggedIn();
checkSessionTimeout();

$database = new Database();
$db = $database->getConnection();
$task = new Task($db);
$user = new User($db);

// Get all users for assignment
try {
    $users = $user->getAllUsers();
} catch (Exception $e) {
    error_log("Submit task page error: " . $e->getMessage());
    $_SESSION['error'] = 'Unable to load users. Please try again.';
    $users = [];
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $task->title = trim($_POST['title'] ?? '');
    $task->description = trim($_POST['description'] ?? '');
    $task->assigned_to = $_POST['assigned_to'] ?? '';
    $task->priority = $_POST['priority'] ?? 'medium';
    $task->due_date = $_POST['due_date'] ?? '';
    $task->submitted_by = $_SESSION['user_id'];
    
    // If admin is submitting, auto-approve
    if ($_SESSION['user_role'] === 'admin') {
        $task->assigned_by = $_SESSION['user_id'];
    }
    
    try {
        if ($task->create()) {
            $_SESSION['success'] = 'Task submitted successfully! ' . 
                ($_SESSION['user_role'] === 'admin' ? 'Task is approved and assigned.' : 'Waiting for admin approval.');
            header('Location: ' . ($_SESSION['user_role'] === 'admin' ? '../admin/tasks.php' : 'my-tasks.php'));
            exit;
        } else {
            $_SESSION['error'] = 'Failed to submit task. Please try again.';
        }
    } catch (Exception $e) {
        $_SESSION['error'] = $e->getMessage();
    }
}

// Set default due date to tomorrow
$default_due_date = date('Y-m-d', strtotime('+1 day'));
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Submit New Task - Task Management System</title>
    <link rel="stylesheet" href="../../assets/css/style.css">
</head>
<body>
    <div class="app-container">
        <?php include '../../includes/sidebar.php'; ?>
        
        <div class="main-content">
            <div class="main-header">
                <div class="header-content">
                    <h1>Submit New Task</h1>
                    <div class="header-actions">
                        <a href="my-tasks.php" class="btn btn-secondary">
                            <span class="btn-icon">←</span>
                            Back to My Tasks
                        </a>
                    </div>
                </div>
            </div>

            <?php if (isset($_SESSION['success'])): ?>
                <div class="alert alert-success">
                    <div class="alert-content">
                        <span class="alert-icon">✅</span>
                        <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (isset($_SESSION['error'])): ?>
                <div class="alert alert-error">
                    <div class="alert-content">
                        <span class="alert-icon">❌</span>
                        <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <div class="card">
                <div class="card-header">
                    <h2>📝 Task Information</h2>
                    <p class="card-subtitle">Fill in the details below to submit a new task</p>
                </div>
                <div class="card-body">
                    <form method="POST" class="task-form" id="taskForm">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="title" class="required">Task Title</label>
                                <input type="text" id="title" name="title" class="form-control" 
                                       placeholder="Enter a clear and descriptive task title" 
                                       required maxlength="255"
                                       value="<?php echo htmlspecialchars($_POST['title'] ?? ''); ?>">
                                <small class="form-text">Be specific about what needs to be done</small>
                            </div>
                            <div class="form-group">
                                <label for="assigned_to" class="required">Assign To</label>
                                <select id="assigned_to" name="assigned_to" class="form-control" required>
                                    <option value="<?php echo $_SESSION['user_id']; ?>">
                                        👤 Myself
                                    </option>
                                    <?php if ($users): ?>
                                        <?php while ($user_item = $users->fetch(PDO::FETCH_ASSOC)): ?>
                                            <?php if ($user_item['id'] != $_SESSION['user_id'] && $user_item['is_active']): ?>
                                                <option value="<?php echo $user_item['id']; ?>" 
                                                    <?php echo ($_POST['assigned_to'] ?? '') == $user_item['id'] ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($user_item['name']); ?>
                                                    (<?php echo ucfirst($user_item['role']); ?>)
                                                </option>
                                            <?php endif; ?>
                                        <?php endwhile; ?>
                                    <?php endif; ?>
                                </select>
                                <small class="form-text">Choose who should complete this task</small>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="description" class="required">Task Description</label>
                            <textarea id="description" name="description" class="form-control" 
                                      placeholder="Provide detailed information about the task, including any specific requirements, steps, or context..." 
                                      required rows="6"><?php echo htmlspecialchars($_POST['description'] ?? ''); ?></textarea>
                            <div class="form-text-row">
                                <small class="form-text">Be as detailed as possible to ensure clarity</small>
                                <small class="form-text char-count">
                                    <span id="charCount">0</span> / 2000 characters
                                </small>
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="due_date" class="required">Due Date</label>
                                <input type="date" id="due_date" name="due_date" class="form-control" 
                                       required min="<?php echo date('Y-m-d'); ?>"
                                       value="<?php echo $_POST['due_date'] ?? $default_due_date; ?>">
                                <small class="form-text">When should this task be completed?</small>
                            </div>
                            <div class="form-group">
                                <label for="priority" class="required">Priority Level</label>
                                <select id="priority" name="priority" class="form-control" required>
                                    <option value="low" <?php echo ($_POST['priority'] ?? 'medium') === 'low' ? 'selected' : ''; ?>>
                                        🟢 Low Priority
                                    </option>
                                    <option value="medium" <?php echo ($_POST['priority'] ?? 'medium') === 'medium' ? 'selected' : ''; ?>>
                                        🟡 Medium Priority
                                    </option>
                                    <option value="high" <?php echo ($_POST['priority'] ?? 'medium') === 'high' ? 'selected' : ''; ?>>
                                        🔴 High Priority
                                    </option>
                                </select>
                                <small class="form-text">How urgent is this task?</small>
                            </div>
                        </div>
                        
                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary btn-large">
                                <span class="btn-icon">📨</span>
                                Submit Task
                            </button>
                            <a href="my-tasks.php" class="btn btn-secondary">Cancel</a>
                        </div>
                        
                        <?php if ($_SESSION['user_role'] !== 'admin'): ?>
                            <div class="form-note">
                                <div class="note-icon">ℹ️</div>
                                <div class="note-content">
                                    <strong>Approval Process:</strong> All tasks submitted by regular users require admin approval before they appear in the task list and are assigned to team members.
                                </div>
                            </div>
                        <?php else: ?>
                            <div class="form-note">
                                <div class="note-icon">⚡</div>
                                <div class="note-content">
                                    <strong>Admin Privilege:</strong> Tasks created by administrators are automatically approved and immediately assigned.
                                </div>
                            </div>
                        <?php endif; ?>
                    </form>
                </div>
            </div>

            <!-- Form Guidelines -->
            <div class="card">
                <div class="card-header">
                    <h3>📋 Submission Guidelines</h3>
                </div>
                <div class="card-body">
                    <div class="guidelines-list">
                        <div class="guideline-item">
                            <span class="guideline-icon">🎯</span>
                            <div class="guideline-content">
                                <strong>Clear Titles</strong>
                                <p>Use specific, action-oriented titles that clearly describe the task</p>
                            </div>
                        </div>
                        <div class="guideline-item">
                            <span class="guideline-icon">📝</span>
                            <div class="guideline-content">
                                <strong>Detailed Descriptions</strong>
                                <p>Include all necessary information, requirements, and context for the assignee</p>
                            </div>
                        </div>
                        <div class="guideline-item">
                            <span class="guideline-icon">⏱️</span>
                            <div class="guideline-content">
                                <strong>Realistic Deadlines</strong>
                                <p>Set achievable due dates considering complexity and workload</p>
                            </div>
                        </div>
                        <div class="guideline-item">
                            <span class="guideline-icon">🚨</span>
                            <div class="guideline-content">
                                <strong>Appropriate Priority</strong>
                                <p>Use high priority only for time-sensitive or critical tasks</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <style>
    .required::after {
        content: " *";
        color: var(--danger);
    }

    .form-text-row {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-top: 0.5rem;
    }

    .char-count {
        color: var(--text-muted);
        font-size: 0.8rem;
    }

    .btn-large {
        padding: 1rem 2rem;
        font-size: 1.1rem;
    }

    .form-note {
        background: var(--bg-secondary);
        padding: 1rem;
        border-radius: var(--radius);
        border-left: 4px solid var(--primary);
        margin-top: 1.5rem;
        display: flex;
        gap: 0.75rem;
        align-items: flex-start;
    }

    .note-icon {
        font-size: 1.2rem;
        flex-shrink: 0;
    }

    .note-content {
        flex: 1;
    }

    .note-content strong {
        color: var(--text-primary);
    }

    .guidelines-list {
        display: flex;
        flex-direction: column;
        gap: 1rem;
    }

    .guideline-item {
        display: flex;
        gap: 1rem;
        align-items: flex-start;
        padding: 1rem;
        background: var(--bg-tertiary);
        border-radius: var(--radius);
        transition: all 0.3s ease;
    }

    .guideline-item:hover {
        background: var(--bg-secondary);
        transform: translateX(5px);
    }

    .guideline-icon {
        font-size: 1.5rem;
        flex-shrink: 0;
    }

    .guideline-content strong {
        color: var(--text-primary);
        display: block;
        margin-bottom: 0.25rem;
    }

    .guideline-content p {
        color: var(--text-secondary);
        margin: 0;
        font-size: 0.9rem;
    }

    @media (max-width: 768px) {
        .form-row {
            grid-template-columns: 1fr;
        }
        
        .guideline-item {
            flex-direction: column;
            text-align: center;
        }
    }
    </style>

    <script>
    // Character count for description
    const description = document.getElementById('description');
    const charCount = document.getElementById('charCount');

    description.addEventListener('input', function() {
        const count = this.value.length;
        charCount.textContent = count;
        
        if (count > 1800) {
            charCount.style.color = 'var(--warning)';
        } else if (count > 1900) {
            charCount.style.color = 'var(--danger)';
        } else {
            charCount.style.color = 'var(--text-muted)';
        }
    });

    // Initialize character count
    charCount.textContent = description.value.length;

    // Form validation
    document.getElementById('taskForm').addEventListener('submit', function(e) {
        const title = document.getElementById('title').value.trim();
        const description = document.getElementById('description').value.trim();
        const dueDate = new Date(document.getElementById('due_date').value);
        const today = new Date();
        today.setHours(0, 0, 0, 0);

        // Basic validation
        if (!title) {
            e.preventDefault();
            showError('Please enter a task title.');
            document.getElementById('title').focus();
            return false;
        }

        if (!description) {
            e.preventDefault();
            showError('Please enter a task description.');
            document.getElementById('description').focus();
            return false;
        }

        if (dueDate < today) {
            e.preventDefault();
            showError('Due date cannot be in the past. Please select a future date.');
            document.getElementById('due_date').focus();
            return false;
        }

        // Show loading state
        const submitBtn = this.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<span class="loading-spinner"></span> Submitting...';
        submitBtn.disabled = true;

        // Re-enable after 3 seconds (in case submission fails)
        setTimeout(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }, 3000);
    });

    function showError(message) {
        // Create temporary error alert
        const errorAlert = document.createElement('div');
        errorAlert.className = 'alert alert-error';
        errorAlert.innerHTML = `
            <div class="alert-content">
                <span class="alert-icon">❌</span>
                ${message}
            </div>
        `;
        
        document.querySelector('.main-content').insertBefore(errorAlert, document.querySelector('.card'));
        
        // Auto-remove after 5 seconds
        setTimeout(() => {
            errorAlert.style.opacity = '0';
            setTimeout(() => errorAlert.remove(), 500);
        }, 5000);
    }

    // Auto-hide existing alerts
    setTimeout(() => {
        const alerts = document.querySelectorAll('.alert');
        alerts.forEach(alert => {
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.style.display = 'none';
            }, 500);
        });
    }, 5000);

    // Add input validation styling
    document.querySelectorAll('input, textarea, select').forEach(input => {
        input.addEventListener('blur', function() {
            if (this.hasAttribute('required') && !this.value.trim()) {
                this.style.borderColor = 'var(--danger)';
            } else {
                this.style.borderColor = '';
            }
        });

        input.addEventListener('input', function() {
            if (this.value.trim()) {
                this.style.borderColor = 'var(--success)';
            }
        });
    });
    </script>
</body>
</html>