<?php
// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

function redirectIfNotLoggedIn() {
    if (!isset($_SESSION['user_id'])) {
        // Store the current URL for redirecting back after login
        $_SESSION['redirect_url'] = $_SERVER['REQUEST_URI'];
        
        // Redirect to login page
        header('Location: /task-management-system/index.php');
        exit;
    }
}

function redirectIfNotAdmin() {
    if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
        $_SESSION['error'] = 'Access denied. Admin privileges required.';
        
        if (isset($_SESSION['user_id'])) {
            // User is logged in but not admin - redirect to user dashboard
            header('Location: /task-management-system/dashboard/user/');
        } else {
            // User is not logged in - redirect to login
            $_SESSION['redirect_url'] = $_SERVER['REQUEST_URI'];
            header('Location: /task-management-system/index.php');
        }
        exit;
    }
}

function isAdmin() {
    return isset($_SESSION['user_id']) && $_SESSION['user_role'] === 'admin';
}

function isLoggedIn() {
    return isset($_SESSION['user_id']);
}

function checkSessionTimeout() {
    $timeout_duration = 3600; // 1 hour in seconds
    
    if (isset($_SESSION['login_time'])) {
        $elapsed_time = time() - $_SESSION['login_time'];
        
        if ($elapsed_time > $timeout_duration) {
            // Session expired
            session_unset();
            session_destroy();
            
            // Set session expired message
            $_SESSION['session_expired'] = true;
            
            header('Location: /task-management-system/index.php?session=expired');
            exit;
        } else {
            // Update login time to extend session
            $_SESSION['login_time'] = time();
        }
    }
}

function sanitizeInput($data) {
    if (is_array($data)) {
        return array_map('sanitizeInput', $data);
    }
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

function validateDate($date, $format = 'Y-m-d') {
    $d = DateTime::createFromFormat($format, $date);
    return $d && $d->format($format) === $date;
}

function redirectToDashboard() {
    if (isset($_SESSION['user_id'])) {
        if ($_SESSION['user_role'] === 'admin') {
            header('Location: /task-management-system/dashboard/admin/');
        } else {
            header('Location: /task-management-system/dashboard/user/');
        }
        exit;
    }
}

function handleUnauthorizedAccess() {
    if (!isset($_SESSION['user_id'])) {
        $_SESSION['error'] = 'Please log in to access this page.';
        header('Location: /task-management-system/index.php');
        exit;
    } else {
        $_SESSION['error'] = 'You do not have permission to access this page.';
        if ($_SESSION['user_role'] === 'admin') {
            header('Location: /task-management-system/dashboard/admin/');
        } else {
            header('Location: /task-management-system/dashboard/user/');
        }
        exit;
    }
}

// Function to get base URL
function getBaseUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $script = $_SERVER['SCRIPT_NAME'];
    
    // Remove the filename from the script path
    $path = dirname($script);
    
    return $protocol . '://' . $host . $path;
}
?>