<?php
// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Include database and user model
include_once 'config/database.php';
include_once 'models/User.php';

// Redirect if already logged in
if (isset($_SESSION['user_id'])) {
    if ($_SESSION['user_role'] === 'admin') {
        header('Location: dashboard/admin/');
    } else {
        header('Location: dashboard/user/');
    }
    exit;
}

$database = new Database();
$db = $database->getConnection();
$user = new User($db);

$error = '';
$email = '';

// Check for session expired message
if (isset($_GET['session']) && $_GET['session'] === 'expired') {
    $error = 'Your session has expired. Please log in again.';
}

// Check for logout success message
if (isset($_SESSION['logout_success'])) {
    $success = $_SESSION['logout_success'];
    unset($_SESSION['logout_success']);
}

// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    
    if (!empty($email) && !empty($password)) {
        // Basic email validation
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = 'Please enter a valid email address.';
        } else {
            try {
                $user_data = $user->getUserByEmail($email);
                
                if ($user_data && password_verify($password, $user_data['password'])) {
                    if ($user_data['is_active']) {
                        // Regenerate session ID for security
                        session_regenerate_id(true);
                        
                        // Set session variables
                        $_SESSION['user_id'] = $user_data['id'];
                        $_SESSION['user_name'] = $user_data['name'];
                        $_SESSION['user_email'] = $user_data['email'];
                        $_SESSION['user_role'] = $user_data['role'];
                        $_SESSION['login_time'] = time();
                        
                        // Log successful login
                        error_log("Successful login: {$user_data['email']} ({$user_data['role']})");
                        
                        // Check if there's a redirect URL
                        $redirect_url = $_SESSION['redirect_url'] ?? null;
                        
                        if ($redirect_url) {
                            // Clear the redirect URL
                            unset($_SESSION['redirect_url']);
                            header('Location: ' . $redirect_url);
                        } else {
                            // Redirect based on user role
                            if ($user_data['role'] === 'admin') {
                                header('Location: dashboard/admin/');
                            } else {
                                header('Location: dashboard/user/');
                            }
                        }
                        exit;
                    } else {
                        $error = 'Your account has been deactivated. Please contact administrator.';
                    }
                } else {
                    $error = 'Invalid email or password.';
                    // Log failed login attempt
                    error_log("Failed login attempt for email: {$email}");
                }
            } catch (Exception $e) {
                error_log("Login error: " . $e->getMessage());
                $error = 'An error occurred during login. Please try again.';
            }
        }
    } else {
        $error = 'Please fill in all fields.';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Task Management System</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .login-body {
            background: linear-gradient(135deg, var(--bg-primary) 0%, var(--bg-secondary) 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 1rem;
            position: relative;
            overflow: hidden;
        }

        .login-body::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, var(--primary) 0%, transparent 70%);
            opacity: 0.05;
            animation: float 20s infinite linear;
        }

        @keyframes float {
            0% { transform: translate(0, 0) rotate(0deg); }
            100% { transform: translate(-50px, -50px) rotate(360deg); }
        }

        .login-container {
            background: var(--bg-card);
            padding: 2.5rem;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-lg);
            width: 100%;
            max-width: 440px;
            border: 1px solid var(--border);
            backdrop-filter: blur(10px);
            position: relative;
            z-index: 1;
        }
        
        .login-header {
            text-align: center;
            margin-bottom: 2rem;
        }
        
        .login-header h1 {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            margin-bottom: 0.5rem;
            font-size: 2rem;
            font-weight: 700;
        }
        
        .login-header p {
            color: var(--text-secondary);
            font-size: 1rem;
        }

        .login-logo {
            font-size: 3rem;
            margin-bottom: 1rem;
        }
        
        .demo-credentials {
            background: var(--bg-secondary);
            padding: 1.25rem;
            border-radius: var(--radius);
            border-left: 4px solid var(--primary);
            margin-top: 1.5rem;
        }
        
        .demo-credentials h4 {
            margin-bottom: 0.75rem;
            color: var(--text-primary);
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .demo-credentials p {
            margin: 0.5rem 0;
            font-size: 0.85rem;
            color: var(--text-secondary);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .credential-role {
            background: var(--primary);
            color: white;
            padding: 0.25rem 0.5rem;
            border-radius: var(--radius-sm);
            font-size: 0.75rem;
            font-weight: 600;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: var(--text-primary);
            font-size: 0.9rem;
        }

        .form-control {
            width: 100%;
            padding: 0.875rem;
            background: var(--bg-secondary);
            border: 2px solid var(--border);
            border-radius: var(--radius);
            color: var(--text-primary);
            font-size: 1rem;
            transition: all 0.3s ease;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
            background: var(--bg-tertiary);
        }

        .btn-login {
            width: 100%;
            padding: 1rem;
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            color: white;
            border: none;
            border-radius: var(--radius);
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 0.5rem;
        }

        .btn-login:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.3);
        }

        .btn-login:active {
            transform: translateY(0);
        }

        .alert {
            padding: 1rem;
            border-radius: var(--radius);
            margin-bottom: 1.5rem;
            border-left: 4px solid;
            animation: slideIn 0.3s ease;
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .alert-success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
            border-color: var(--success);
        }

        .alert-error {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
            border-color: var(--danger);
        }

        .password-container {
            position: relative;
        }

        .toggle-password {
            position: absolute;
            right: 12px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: var(--text-muted);
            cursor: pointer;
            padding: 0.25rem;
        }

        .toggle-password:hover {
            color: var(--text-primary);
        }

        @media (max-width: 480px) {
            .login-container {
                padding: 2rem 1.5rem;
            }
            
            .login-header h1 {
                font-size: 1.75rem;
            }
        }
    </style>
</head>
<body class="login-body">
    <div class="login-container">
        <div class="login-header">
            <div class="login-logo"><img src="./assets/Logo-white-1-2.png" alt="" width="200px"></div>
            <h1>Your Daily Task</h1>
            <p>Streamline your team's workflow</p>
        </div>

        <?php if (isset($success)): ?>
            <div class="alert alert-success">
                <strong>Success!</strong> <?php echo $success; ?>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <strong>Error!</strong> <?php echo $error; ?>
            </div>
        <?php endif; ?>

        <form method="POST" class="login-form">
            <div class="form-group">
                <label for="email">Email Address</label>
                <input type="email" id="email" name="email" class="form-control" 
                       placeholder="Enter your email" required 
                       value="<?php echo htmlspecialchars($email); ?>"
                       autocomplete="email">
            </div>
            
            <div class="form-group">
                <label for="password">Password</label>
                <div class="password-container">
                    <input type="password" id="password" name="password" class="form-control" 
                           placeholder="Enter your password" required
                           autocomplete="current-password">
                    <button type="button" class="toggle-password" onclick="togglePasswordVisibility()">
                        👁️
                    </button>
                </div>
            </div>
            
            <button type="submit" class="btn-login">
                Sign In
            </button>
        </form>

        <!-- Demo Credentials
        <div class="demo-credentials">
            <h4>Demo Credentials</h4>
            <p>
                <span>admin@taskmanager.com</span>
                <span class="credential-role">Admin</span>
            </p>
            <p>
                <span>user@taskmanager.com</span>
                <span class="credential-role">User</span>
            </p>
            <p style="margin-top: 0.75rem; font-size: 0.8rem; color: var(--text-muted);">
                Password for both accounts: <strong>password</strong>
            </p>
        </div> -->
    </div>

    <script>
        // Focus on email field when page loads
        document.getElementById('email').focus();

        // Toggle password visibility
        function togglePasswordVisibility() {
            const passwordInput = document.getElementById('password');
            const toggleButton = document.querySelector('.toggle-password');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                toggleButton.textContent = '🔒';
            } else {
                passwordInput.type = 'password';
                toggleButton.textContent = '👁️';
            }
        }

        // Form validation
        document.querySelector('.login-form').addEventListener('submit', function(e) {
            const email = document.getElementById('email').value.trim();
            const password = document.getElementById('password').value;
            
            if (!email || !password) {
                e.preventDefault();
                showError('Please fill in all fields.');
                return false;
            }
            
            // Basic email validation
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                e.preventDefault();
                showError('Please enter a valid email address.');
                return false;
            }

            // Show loading state
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            submitBtn.innerHTML = '<span class="loading-spinner"></span> Signing In...';
            submitBtn.disabled = true;
        });

        function showError(message) {
            // Create temporary error alert
            const errorAlert = document.createElement('div');
            errorAlert.className = 'alert alert-error';
            errorAlert.innerHTML = `
                <div class="alert-content">
                    <span class="alert-icon">❌</span>
                    ${message}
                </div>
            `;
            
            const loginContainer = document.querySelector('.login-container');
            const existingAlert = loginContainer.querySelector('.alert');
            
            if (existingAlert) {
                existingAlert.remove();
            }
            
            loginContainer.insertBefore(errorAlert, loginContainer.querySelector('form'));
            
            // Auto-remove after 5 seconds
            setTimeout(() => {
                errorAlert.style.opacity = '0';
                setTimeout(() => errorAlert.remove(), 500);
            }, 5000);
        }

        // Auto-hide alerts after 5 seconds
        setTimeout(() => {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                alert.style.opacity = '0';
                alert.style.transition = 'opacity 0.5s ease';
                setTimeout(() => {
                    alert.style.display = 'none';
                }, 500);
            });
        }, 5000);

        // Add enter key support
        document.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                const focused = document.activeElement;
                if (focused && (focused.type === 'email' || focused.type === 'password' || focused.type === 'text')) {
                    document.querySelector('.btn-login').click();
                }
            }
        });
    </script>
</body>
</html>