<?php
class Task {
    private $conn;
    private $table_name = "tasks";

    public $id;
    public $title;
    public $description;
    public $assigned_to;
    public $assigned_by;
    public $submitted_by;
    public $status;
    public $priority;
    public $due_date;
    public $completion_date;
    public $is_approved;
    public $approved_at;
    public $approved_by;
    public $created_at;

    public function __construct($db) {
        $this->conn = $db;
    }

    // Create a new task
    public function create() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET title = :title, 
                      description = :description, 
                      assigned_to = :assigned_to, 
                      assigned_by = :assigned_by,
                      submitted_by = :submitted_by,
                      status = :status, 
                      priority = :priority, 
                      due_date = :due_date,
                      is_approved = :is_approved,
                      created_at = NOW()";
        
        $stmt = $this->conn->prepare($query);

        // Validate required fields
        if (empty($this->title) || empty($this->description) || empty($this->due_date)) {
            throw new Exception("Title, description, and due date are required.");
        }

        // Validate due date
        if (!validateDate($this->due_date)) {
            throw new Exception("Invalid due date format.");
        }

        // Auto-approve if admin is creating, otherwise require approval
        $is_approved = ($this->assigned_by !== null) ? 1 : 0;
        
        // Set default status
        $status = $this->status ?: 'pending';

        $stmt->bindParam(":title", $this->title);
        $stmt->bindParam(":description", $this->description);
        $stmt->bindParam(":assigned_to", $this->assigned_to);
        $stmt->bindParam(":assigned_by", $this->assigned_by);
        $stmt->bindParam(":submitted_by", $this->submitted_by);
        $stmt->bindParam(":status", $status);
        $stmt->bindParam(":priority", $this->priority);
        $stmt->bindParam(":due_date", $this->due_date);
        $stmt->bindParam(":is_approved", $is_approved, PDO::PARAM_BOOL);

        if ($stmt->execute()) {
            $task_id = $this->conn->lastInsertId();
            
            // Log task creation
            $comment = "Task created by " . $this->getUserName($this->submitted_by);
            $this->addComment($task_id, $this->submitted_by, $comment);
            
            return $task_id;
        }
        return false;
    }

    // Read all tasks (admin sees all, users see only their tasks)
    public function readAll($user_id = null, $user_role = 'user') {
        if ($user_role === 'admin') {
            $query = "SELECT t.*, 
                             u.name as assigned_to_name, 
                             u.role as assigned_to_role,
                             u2.name as assigned_by_name, 
                             u3.name as submitted_by_name,
                             u4.name as approved_by_name
                      FROM " . $this->table_name . " t 
                      LEFT JOIN users u ON t.assigned_to = u.id 
                      LEFT JOIN users u2 ON t.assigned_by = u2.id 
                      LEFT JOIN users u3 ON t.submitted_by = u3.id 
                      LEFT JOIN users u4 ON t.approved_by = u4.id 
                      WHERE t.status != 'rejected'
                      ORDER BY 
                        CASE t.priority 
                            WHEN 'high' THEN 1 
                            WHEN 'medium' THEN 2 
                            WHEN 'low' THEN 3 
                        END,
                        t.due_date ASC,
                        t.created_at DESC";
        } else {
            $query = "SELECT t.*, 
                             u.name as assigned_to_name, 
                             u2.name as assigned_by_name, 
                             u3.name as submitted_by_name,
                             u4.name as approved_by_name
                      FROM " . $this->table_name . " t 
                      LEFT JOIN users u ON t.assigned_to = u.id 
                      LEFT JOIN users u2 ON t.assigned_by = u2.id 
                      LEFT JOIN users u3 ON t.submitted_by = u3.id 
                      LEFT JOIN users u4 ON t.approved_by = u4.id 
                      WHERE (t.assigned_to = ? OR t.submitted_by = ?) 
                      AND t.is_approved = 1 
                      AND t.status != 'rejected'
                      ORDER BY 
                        CASE t.priority 
                            WHEN 'high' THEN 1 
                            WHEN 'medium' THEN 2 
                            WHEN 'low' THEN 3 
                        END,
                        t.due_date ASC,
                        t.created_at DESC";
        }

        $stmt = $this->conn->prepare($query);
        if ($user_role !== 'admin') {
            $stmt->bindParam(1, $user_id);
            $stmt->bindParam(2, $user_id);
        }
        $stmt->execute();
        return $stmt;
    }

    // Get single task by ID
    public function getTask($task_id) {
        $query = "SELECT t.*, 
                         u.name as assigned_to_name, 
                         u2.name as assigned_by_name, 
                         u3.name as submitted_by_name,
                         u4.name as approved_by_name
                  FROM " . $this->table_name . " t 
                  LEFT JOIN users u ON t.assigned_to = u.id 
                  LEFT JOIN users u2 ON t.assigned_by = u2.id 
                  LEFT JOIN users u3 ON t.submitted_by = u3.id 
                  LEFT JOIN users u4 ON t.approved_by = u4.id 
                  WHERE t.id = ?
                  LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $task_id);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Get tasks pending approval
    public function getPendingApprovalTasks() {
        $query = "SELECT t.*, 
                         u.name as assigned_to_name, 
                         u2.name as submitted_by_name 
                  FROM " . $this->table_name . " t 
                  LEFT JOIN users u ON t.assigned_to = u.id 
                  LEFT JOIN users u2 ON t.submitted_by = u2.id 
                  WHERE t.is_approved = 0 
                  AND t.status != 'rejected'
                  ORDER BY t.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }

    // Get count of pending approval tasks
    public function getPendingApprovalCount() {
        $query = "SELECT COUNT(*) as count FROM " . $this->table_name . " 
                  WHERE is_approved = 0 AND status != 'rejected'";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['count'];
    }

    // Approve a task
    public function approveTask($task_id, $approved_by) {
        $query = "UPDATE " . $this->table_name . " 
                  SET is_approved = 1, 
                      approved_at = NOW(), 
                      approved_by = :approved_by,
                      status = 'pending'
                  WHERE id = :task_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":task_id", $task_id);
        $stmt->bindParam(":approved_by", $approved_by);
        
        if ($stmt->execute()) {
            // Add approval comment
            $approver_name = $this->getUserName($approved_by);
            $comment = "Task approved by {$approver_name}";
            $this->addComment($task_id, $approved_by, $comment);
            return true;
        }
        return false;
    }

    // Reject a task
    public function rejectTask($task_id) {
        $query = "UPDATE " . $this->table_name . " 
                  SET status = 'rejected'
                  WHERE id = :task_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":task_id", $task_id);
        return $stmt->execute();
    }

    // Update task status
    public function updateStatus($task_id, $status, $comment = null, $user_id = null) {
        // Validate status
        $valid_statuses = ['pending', 'in-progress', 'completed', 'blocked'];
        if (!in_array($status, $valid_statuses)) {
            throw new Exception("Invalid status.");
        }

        $query = "UPDATE " . $this->table_name . " 
                  SET status = :status, 
                      completion_date = :completion_date
                  WHERE id = :task_id";
        
        $stmt = $this->conn->prepare($query);

        $completion_date = ($status === 'completed') ? date('Y-m-d H:i:s') : null;

        $stmt->bindParam(":status", $status);
        $stmt->bindParam(":completion_date", $completion_date);
        $stmt->bindParam(":task_id", $task_id);

        if ($stmt->execute()) {
            // Add comment if provided
            if ($comment && $user_id) {
                $this->addComment($task_id, $user_id, $comment);
            } else {
                // Add automatic status change comment
                $user_name = $this->getUserName($user_id);
                $auto_comment = "Status changed to '{$status}' by {$user_name}";
                $this->addComment($task_id, $user_id, $auto_comment);
            }
            return true;
        }
        return false;
    }

    // Reassign task to different user
    public function reassignTask($task_id, $new_user_id, $reassigned_by) {
        $query = "UPDATE " . $this->table_name . " 
                  SET assigned_to = :new_user_id
                  WHERE id = :task_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":task_id", $task_id);
        $stmt->bindParam(":new_user_id", $new_user_id);
        
        if ($stmt->execute()) {
            // Add comment about reassignment
            $reassigner_name = $this->getUserName($reassigned_by);
            $new_user_name = $this->getUserName($new_user_id);
            $comment = "Task reassigned by {$reassigner_name} to {$new_user_name}";
            return $this->addComment($task_id, $reassigned_by, $comment);
        }
        return false;
    }

    // Update task details
    public function updateTask($task_id, $data) {
        $query = "UPDATE " . $this->table_name . " 
                  SET title = :title,
                      description = :description,
                      assigned_to = :assigned_to,
                      priority = :priority,
                      due_date = :due_date
                  WHERE id = :task_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":title", $data['title']);
        $stmt->bindParam(":description", $data['description']);
        $stmt->bindParam(":assigned_to", $data['assigned_to']);
        $stmt->bindParam(":priority", $data['priority']);
        $stmt->bindParam(":due_date", $data['due_date']);
        $stmt->bindParam(":task_id", $task_id);
        
        return $stmt->execute();
    }

    // Delete task
    public function deleteTask($task_id) {
        // Start transaction
        $this->conn->beginTransaction();

        try {
            // First delete comments
            $delete_comments = "DELETE FROM task_comments WHERE task_id = :task_id";
            $stmt1 = $this->conn->prepare($delete_comments);
            $stmt1->bindParam(":task_id", $task_id);
            $stmt1->execute();

            // Then delete the task
            $query = "DELETE FROM " . $this->table_name . " WHERE id = :task_id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(":task_id", $task_id);
            $stmt->execute();

            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            throw $e;
        }
    }

    // Add comment to task
    public function addComment($task_id, $user_id, $comment) {
        if (empty($comment)) {
            throw new Exception("Comment cannot be empty.");
        }

        $query = "INSERT INTO task_comments 
                  SET task_id = :task_id, 
                      user_id = :user_id, 
                      comment = :comment,
                      created_at = NOW()";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":task_id", $task_id);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->bindParam(":comment", $comment);
        
        return $stmt->execute();
    }

    // Get task comments
    public function getComments($task_id) {
        $query = "SELECT tc.*, u.name as user_name 
                  FROM task_comments tc 
                  LEFT JOIN users u ON tc.user_id = u.id 
                  WHERE tc.task_id = ? 
                  ORDER BY tc.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $task_id);
        $stmt->execute();
        return $stmt;
    }

    // Get user submitted tasks
    public function getUserSubmittedTasks($user_id) {
        $query = "SELECT t.*, 
                         u.name as assigned_to_name, 
                         u2.name as assigned_by_name 
                  FROM " . $this->table_name . " t 
                  LEFT JOIN users u ON t.assigned_to = u.id 
                  LEFT JOIN users u2 ON t.assigned_by = u2.id 
                  WHERE t.submitted_by = ? 
                  AND t.status != 'rejected'
                  ORDER BY t.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $user_id);
        $stmt->execute();
        return $stmt;
    }

    // Get user tasks
    public function getUserTasks($user_id) {
        $query = "SELECT t.*, u.name as assigned_by_name 
                  FROM " . $this->table_name . " t 
                  LEFT JOIN users u ON t.assigned_by = u.id 
                  WHERE t.assigned_to = ? AND t.is_approved = 1 
                  ORDER BY 
                    CASE t.priority 
                        WHEN 'high' THEN 1 
                        WHEN 'medium' THEN 2 
                        WHEN 'low' THEN 3 
                    END,
                    t.due_date ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $user_id);
        $stmt->execute();
        return $stmt;
    }

    // Get user tasks by status
    public function getUserTasksByStatus($user_id, $status) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE assigned_to = ? AND status = ? AND is_approved = 1
                  ORDER BY due_date ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $user_id);
        $stmt->bindParam(2, $status);
        $stmt->execute();
        return $stmt;
    }

    // Get recent user tasks
    public function getRecentUserTasks($user_id, $limit = 5) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE assigned_to = ? AND is_approved = 1 
                  ORDER BY created_at DESC 
                  LIMIT ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $user_id);
        $stmt->bindParam(2, $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt;
    }

    // Get recent tasks (for admin)
    public function getRecentTasks($limit = 5) {
        $query = "SELECT t.*, u.name as assigned_to_name 
                  FROM " . $this->table_name . " t 
                  LEFT JOIN users u ON t.assigned_to = u.id 
                  WHERE t.is_approved = 1 
                  ORDER BY t.created_at DESC 
                  LIMIT ?";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt;
    }

    // Get task statistics
    public function getTaskStatistics() {
        $query = "SELECT 
                  COUNT(*) as total_tasks,
                  SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_tasks,
                  SUM(CASE WHEN status = 'in-progress' THEN 1 ELSE 0 END) as in_progress_tasks,
                  SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_tasks,
                  SUM(CASE WHEN status = 'blocked' THEN 1 ELSE 0 END) as blocked_tasks
                  FROM " . $this->table_name . " 
                  WHERE is_approved = 1 AND status != 'rejected'";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Get user task count
    public function getUserTaskCount($user_id) {
        $query = "SELECT COUNT(*) as task_count 
                  FROM " . $this->table_name . " 
                  WHERE assigned_to = :user_id 
                  AND is_approved = 1 
                  AND status != 'rejected'";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['task_count'];
    }

    // Get tasks for export
    public function getTasksForExport() {
        $query = "SELECT t.*, 
                         u.name as assigned_to_name, 
                         u2.name as assigned_by_name, 
                         u3.name as submitted_by_name 
                  FROM " . $this->table_name . " t 
                  LEFT JOIN users u ON t.assigned_to = u.id 
                  LEFT JOIN users u2 ON t.assigned_by = u2.id 
                  LEFT JOIN users u3 ON t.submitted_by = u3.id 
                  WHERE t.is_approved = 1 
                  AND t.status != 'rejected'
                  ORDER BY t.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }

    // Get overdue tasks
    public function getOverdueTasks() {
        $query = "SELECT t.*, 
                         u.name as assigned_to_name, 
                         u2.name as assigned_by_name 
                  FROM " . $this->table_name . " t 
                  LEFT JOIN users u ON t.assigned_to = u.id 
                  LEFT JOIN users u2 ON t.assigned_by = u2.id 
                  WHERE t.due_date < CURDATE() 
                  AND t.status NOT IN ('completed', 'rejected')
                  AND t.is_approved = 1
                  ORDER BY t.due_date ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }

    // Get tasks by status
    public function getTasksByStatus($status) {
        $query = "SELECT t.*, 
                         u.name as assigned_to_name, 
                         u2.name as assigned_by_name 
                  FROM " . $this->table_name . " t 
                  LEFT JOIN users u ON t.assigned_to = u.id 
                  LEFT JOIN users u2 ON t.assigned_by = u2.id 
                  WHERE t.status = :status 
                  AND t.is_approved = 1
                  ORDER BY t.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":status", $status);
        $stmt->execute();
        return $stmt;
    }

    // Get tasks by priority
    public function getTasksByPriority($priority) {
        $query = "SELECT t.*, 
                         u.name as assigned_to_name, 
                         u2.name as assigned_by_name 
                  FROM " . $this->table_name . " t 
                  LEFT JOIN users u ON t.assigned_to = u.id 
                  LEFT JOIN users u2 ON t.assigned_by = u2.id 
                  WHERE t.priority = :priority 
                  AND t.is_approved = 1
                  AND t.status != 'rejected'
                  ORDER BY t.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":priority", $priority);
        $stmt->execute();
        return $stmt;
    }

    // Helper method to get user name
    private function getUserName($user_id) {
        if (!$user_id) return 'System';
        
        $query = "SELECT name FROM users WHERE id = ? LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $user_id);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['name'] : 'Unknown User';
    }
}

// Include helper functions if not already included
if (!function_exists('validateDate')) {
    function validateDate($date, $format = 'Y-m-d') {
        $d = DateTime::createFromFormat($format, $date);
        return $d && $d->format($format) === $date;
    }
}
?>