<?php
class User {
    private $conn;
    private $table_name = "users";

    public $id;
    public $name;
    public $email;
    public $password;
    public $role;
    public $is_active;
    public $created_at;

    public function __construct($db) {
        $this->conn = $db;
    }

    // Create new user
    public function create() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET name = :name, 
                      email = :email, 
                      password = :password, 
                      role = :role,
                      is_active = 1,
                      created_at = NOW()";
        
        $stmt = $this->conn->prepare($query);

        // Validate input
        if (empty($this->name) || empty($this->email) || empty($this->password)) {
            throw new Exception("All fields are required.");
        }

        if (!filter_var($this->email, FILTER_VALIDATE_EMAIL)) {
            throw new Exception("Invalid email format.");
        }

        if ($this->emailExists($this->email)) {
            throw new Exception("Email already exists.");
        }

        $hashed_password = password_hash($this->password, PASSWORD_DEFAULT);

        $stmt->bindParam(":name", $this->name);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":password", $hashed_password);
        $stmt->bindParam(":role", $this->role);

        if ($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        return false;
    }

    // Get user by email
    public function getUserByEmail($email) {
        $query = "SELECT id, name, email, password, role, is_active, created_at
                  FROM " . $this->table_name . " 
                  WHERE email = ? AND is_active = 1
                  LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $email);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Get user by ID
    public function getUserById($user_id) {
        $query = "SELECT id, name, email, role, is_active, created_at 
                  FROM " . $this->table_name . " 
                  WHERE id = ?
                  LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $user_id);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Get all users
    public function getAllUsers() {
        $query = "SELECT id, name, email, role, is_active, created_at 
                  FROM " . $this->table_name . " 
                  ORDER BY created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt;
    }

    // Get total users count
    public function getTotalUsersCount() {
        $query = "SELECT COUNT(*) as count FROM " . $this->table_name;
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['count'];
    }

    // Update user status
    public function updateUserStatus($user_id, $status) {
        $query = "UPDATE " . $this->table_name . " 
                  SET is_active = :status 
                  WHERE id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $user_id);
        $stmt->bindParam(":status", $status, PDO::PARAM_BOOL);
        return $stmt->execute();
    }

    // Delete user
    public function deleteUser($user_id) {
        // Check if user exists
        $user = $this->getUserById($user_id);
        if (!$user) {
            throw new Exception("User not found.");
        }

        // Don't allow deleting your own account
        if ($user_id == ($_SESSION['user_id'] ?? null)) {
            throw new Exception("Cannot delete your own account.");
        }

        // Don't allow deleting admin accounts
        if ($user['role'] === 'admin') {
            throw new Exception("Cannot delete admin accounts.");
        }

        // Start transaction
        $this->conn->beginTransaction();

        try {
            // Set assigned_to to NULL for user's tasks
            $reassign_query = "UPDATE tasks SET assigned_to = NULL WHERE assigned_to = :user_id";
            $stmt1 = $this->conn->prepare($reassign_query);
            $stmt1->bindParam(":user_id", $user_id);
            $stmt1->execute();
            
            // Delete the user
            $query = "DELETE FROM " . $this->table_name . " WHERE id = :user_id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(":user_id", $user_id);
            $stmt->execute();

            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            $this->conn->rollBack();
            throw $e;
        }
    }

    // Update user profile
    public function updateProfile($user_id, $data) {
        $query = "UPDATE " . $this->table_name . " 
                  SET name = :name, 
                      email = :email 
                  WHERE id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":name", $data['name']);
        $stmt->bindParam(":email", $data['email']);
        $stmt->bindParam(":user_id", $user_id);
        
        return $stmt->execute();
    }

    // Change password
    public function changePassword($user_id, $new_password) {
        if (strlen($new_password) < 6) {
            throw new Exception("Password must be at least 6 characters long.");
        }

        $query = "UPDATE " . $this->table_name . " 
                  SET password = :password 
                  WHERE id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
        $stmt->bindParam(":password", $hashed_password);
        $stmt->bindParam(":user_id", $user_id);
        
        return $stmt->execute();
    }

    // Check if email exists
    public function emailExists($email) {
        $query = "SELECT id FROM " . $this->table_name . " WHERE email = ? LIMIT 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $email);
        $stmt->execute();
        
        return $stmt->rowCount() > 0;
    }

    // Get users statistics
    public function getUsersStatistics() {
        $query = "SELECT 
                  COUNT(*) as total_users,
                  SUM(CASE WHEN role = 'admin' THEN 1 ELSE 0 END) as admin_users,
                  SUM(CASE WHEN role = 'user' THEN 1 ELSE 0 END) as regular_users,
                  SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_users,
                  SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive_users
                  FROM " . $this->table_name;
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
?>